/*
 *  linux/arch/arm/mach-dmw/pm.c
 *
 *  Copyright (C) 2011 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/clk.h>
#include <linux/clkdev.h>
#include <linux/cpuidle.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/pm.h>
#include <linux/suspend.h>
#include <mach/powerdomain.h>
#include <mach/system.h>

#include "irq.h"
#include "pm.h"

#define param_lowpower_external_cnt_bits 16
#define param_lowpower_external_cnt_reg  21
#define param_lowpower_external_cnt_offset 16
#define param_lowpower_internal_cnt_bits 16
#define param_lowpower_internal_cnt_reg  22
#define param_lowpower_internal_cnt_offset 8
#define param_lowpower_power_down_cnt_bits 16
#define param_lowpower_power_down_cnt_reg  20
#define param_lowpower_power_down_cnt_offset 16
#define param_lowpower_self_refresh_cnt_bits 16
#define param_lowpower_self_refresh_cnt_reg  21
#define param_lowpower_self_refresh_cnt_offset 0

/* defined by DSPG */
#define param_dram_type_bits 4
#define param_dram_type_reg  129
#define param_dram_type_offset 16

/* denali parameter access */
#define denali_readl(_regno)		readl(denali_base + (_regno) * 4)
#define denali_writel(_val, _regno)	writel((_val), denali_base + (_regno) * 4)

#define denali_set(_name, _val)                               \
({                                                            \
	uint32_t msk = (1 << param_##_name##_bits) - 1;       \
	uint32_t reg;                                         \
	                                                      \
	reg = denali_readl(param_##_name##_reg);              \
	reg &= ~(msk << param_##_name##_offset);              \
	reg |= (_val) << param_##_name##_offset;              \
	denali_writel(reg, param_##_name##_reg);              \
})

#define denali_get(_name)                                     \
({                                                            \
	uint32_t msk = (1 << param_##_name##_bits) - 1;       \
	uint32_t reg;                                         \
	                                                      \
	reg = denali_readl(param_##_name##_reg);              \
	reg >>= param_##_name##_offset;                       \
	reg & msk;                                            \
})

#define cmu_readl(_reg)		readl(IO_ADDRESS(DMW_CMU_BASE) + (_reg))
#define cmu_writel(_val, _reg)	writel((_val), IO_ADDRESS(DMW_CMU_BASE) + (_reg))

static void *denali_base;
static struct clk *cpu_clk;
static struct clk *pll3_clk;
static struct clk *sysclk_clk;
static int quirks;
static unsigned overdrive_gpio = -1;
static unsigned dram_rtt_gpio = -1;
static struct cpuidle_device *dmw_idle_dev;

static int dmw_sleep(int req_mode, int safe_mode)
{
	unsigned long reg;
	int mode;
	struct clk *cpu_fast_clk = clk_get_parent(cpu_clk);

	req_mode |= quirks;
	safe_mode |= quirks;

	/*
	 * Stay here until an IRQ is pending. FIQ's will wake us up too and
	 * will be processed but they will not cause us to leave.
	 */
	while (!dmw_irq_pending()) {
		int active_io, active_bm;
		unsigned long swclkenr1 = 0;
		local_fiq_disable();

		active_io = sysclk_clk->iocnt > 0;
		active_bm = atomic_read(&dmw_active_bm_cnt) > 0;

		/* Enable automatic system clock control if possible */
		if (!active_io && !active_bm) {
			reg = cmu_readl(DMW_CMU_CPUCLKCNTRL);
			reg |= 1 << 18;
			cmu_writel(reg, DMW_CMU_CPUCLKCNTRL);

			/*
			 * Switch off uarts as they anyways don't work reliably
			 * from here on.
			 */
			swclkenr1 = cmu_readl(DMW_CMU_SWCLKENR1);
			cmu_writel(0x90, DMW_CMU_WRPR);
			cmu_writel(swclkenr1 & ~(0x7 << 6), DMW_CMU_SWCLKENR1);
		}

		/* Fall back to safe mode if bus masters interfere with chosen mode */
		if (req_mode & SLEEP_CHECK_BM && active_bm)
			mode = safe_mode;
		else
			mode = req_mode;

		/* Make sure we don't disable PLLs that are needed by someone
		 * else */
		if (cpu_fast_clk->refcnt > 1)
			mode &= ~__SLEEP_PLL2_OFF;
		if (pll3_clk->refcnt > 1)
			mode &= ~__SLEEP_PLL3_OFF;

		dmw_sleep_selfrefresh(mode, denali_base, overdrive_gpio, dram_rtt_gpio);

		/* Disable automatic system clock control if enabled previously */
		if (!active_io && !active_bm) {
			/* restore UART clocks */
			do {
				cmu_writel(0x90, DMW_CMU_WRPR);
				cmu_writel(swclkenr1, DMW_CMU_SWCLKENR1);
			} while (cmu_readl(DMW_CMU_SWCLKENR1) != swclkenr1);

			reg = cmu_readl(DMW_CMU_CPUCLKCNTRL);
			reg &= ~(1 << 18);
			cmu_writel(reg, DMW_CMU_CPUCLKCNTRL);
		}

		local_fiq_enable();
	}

	return 0;
}

static int dmw_idle_enter(struct cpuidle_device *dev, struct cpuidle_state *state)
{
	struct timespec ts_preidle, ts_postidle, ts_idle;

	/* Fall back to safe idle mode if DRAM access is required */
	if ((int)state->driver_data & SLEEP_CHECK_BM &&
	    atomic_read(&dmw_active_bm_cnt) > 0) {
		dev->last_state = dev->safe_state;
		return dev->safe_state->enter(dev, dev->safe_state);
	}

	/* Used to keep track of the total time in idle */
	getnstimeofday(&ts_preidle);

	dmw_sleep((int)state->driver_data, (int)dev->safe_state->driver_data);

	/* calculate sleep time */
	getnstimeofday(&ts_postidle);

	local_irq_enable(); /* idle handler must re-enable IRQs on exit */

	ts_idle = timespec_sub(ts_postidle, ts_preidle);
	return ts_idle.tv_nsec / NSEC_PER_USEC + ts_idle.tv_sec * USEC_PER_SEC;
}

struct cpuidle_device dmw_idle_dev_lpddr2 = {
	.states = {
		{
			.name = "C0",
			.desc = "M4",
			.driver_data = (void *)SLEEP_DDR_MODE_4,
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 0, /* in US */
			.power_usage = 150, /* in mW */
			.target_residency = 0, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C1",
			.desc = "M4+PLL2",
			.driver_data = (void *)(SLEEP_DDR_MODE_4 |
						SLEEP_CPU_PLL_OFF),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 200, /* in US */
			.power_usage = 140, /* in mW */
			.target_residency = 400, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C2",
			.desc = "M4+PLL2+PLL3",
			.driver_data = (void *)(SLEEP_DDR_MODE_4 |
						SLEEP_CHECK_BM |
						SLEEP_DDR_PLL_OFF |
						SLEEP_CPU_PLL_OFF),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 200, /* in US */
			.power_usage = 5, /* in mW */
			.target_residency = 400, /* in US */
			.enter = dmw_idle_enter,
		},
	},
	.state_count = 3, /* static array -> cannot use ARRAY_SIZE! */
	.safe_state = &dmw_idle_dev_lpddr2.states[1],
};

struct cpuidle_device dmw_idle_dev_ddr2 = {
	.states = {
		{
			.name = "C0",
			.desc = "M1",
			.driver_data = (void *)SLEEP_DDR_MODE_1,
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 0, /* in US */
			.power_usage = 200, /* in mW */
			.target_residency = 0, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C1",
			.desc = "M4",
			.driver_data = (void *)(SLEEP_DDR_MODE_4 |
						SLEEP_CHECK_BM),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 1, /* in US */
			.power_usage = 150, /* in mW */
			.target_residency = 2, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C2",
			.desc = "M4+PLL2+PLL3",
			.driver_data = (void *)(SLEEP_DDR_MODE_4 |
						SLEEP_CHECK_BM |
						SLEEP_DDR_PLL_OFF |
						SLEEP_CPU_PLL_OFF),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 200, /* in US */
			.power_usage = 50, /* in mW */
			.target_residency = 400, /* in US */
			.enter = dmw_idle_enter,
		},
	},
	.state_count = 3, /* static array -> cannot use ARRAY_SIZE! */
	.safe_state = &dmw_idle_dev_ddr2.states[0],
};

struct cpuidle_device dmw_idle_dev_ddr2_rev_a = {
	.states = {
		{
			.name = "C0",
			.desc = "M1",
			.driver_data = (void *)SLEEP_DDR_MODE_1,
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 0, /* in US */
			.power_usage = 200, /* in mW */
			.target_residency = 0, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C1",
			.desc = "M3",
			.driver_data = (void *)(SLEEP_DDR_MODE_3 |
						SLEEP_CHECK_BM),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 1, /* in US */
			.power_usage = 150, /* in mW */
			.target_residency = 2, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C2",
			.desc = "M3+PLL2",
			.driver_data = (void *)(SLEEP_DDR_MODE_3 |
						SLEEP_CHECK_BM |
						SLEEP_CPU_PLL_OFF),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 200, /* in US */
			.power_usage = 140, /* in mW */
			.target_residency = 400, /* in US */
			.enter = dmw_idle_enter,
		},
	},
	.state_count = 3, /* static array -> cannot use ARRAY_SIZE! */
	.safe_state = &dmw_idle_dev_ddr2_rev_a.states[0],
};

struct cpuidle_driver dmw_idle_driver = {
	.name  = "dmw_idle",
	.owner = THIS_MODULE,
};


static int dmw_pm_prepare(void)
{
	disable_hlt();
	return 0;
}

static int dmw_pm_prepare_late(void)
{
	if (sysclk_clk->iocnt > 0 || atomic_read(&dmw_active_bm_cnt)) {
		printk(KERN_WARNING "%d active IO devices on SYSCLK and"
		       " %d active bus masters remaining!\n",
		       sysclk_clk->iocnt, atomic_read(&dmw_active_bm_cnt));
		dmw_clk_print_active();
	}

	return 0;
}

static int dmw_pm_enter(suspend_state_t state)
{
	int ret = 0;
	struct cpuidle_device *dev = dmw_idle_dev;

	switch (state) {
	case PM_SUSPEND_STANDBY:
	case PM_SUSPEND_MEM:
		dmw_irq_suspend();
		dmw_pwrdom_disable(DMW_PWRDOM_CORTEX_NEON);
		ret = dmw_sleep((int)dev->states[dev->state_count-1].driver_data,
				(int)dev->safe_state->driver_data);
		dmw_pwrdom_enable(DMW_PWRDOM_CORTEX_NEON);
		dmw_irq_resume();
		break;
	default:
		ret = -EINVAL;
	}

	return ret;
}

static void dmw_pm_finish(void)
{
	enable_hlt();
}

static struct platform_suspend_ops dmw_pm_ops = {
	.prepare	= dmw_pm_prepare,
	.prepare_late	= dmw_pm_prepare_late,
	.enter		= dmw_pm_enter,
	.finish		= dmw_pm_finish,
	.valid		= suspend_valid_only_mem,
};

static unsigned long __init ns2clk(unsigned long ns, unsigned long f)
{
	/* Assume that f is in MHz range and ns < 1us */
	return ((f / 1000) * ns) / 1000000 + 1;
}

static void set_lowpower_timings(unsigned long cke, unsigned long self)
{
	denali_set(lowpower_power_down_cnt, cke);
	denali_set(lowpower_self_refresh_cnt, self);
	denali_set(lowpower_external_cnt, self);
	denali_set(lowpower_internal_cnt, self);
}

static int __init dmw_pm_probe(struct platform_device *pdev)
{
	struct dmw_pm_pdata *pdata = pdev->dev.platform_data;
	unsigned long dram_freq;
	int ret = 0;
	unsigned long reg, flags;

	if (pdata) {
		overdrive_gpio = pdata->overdrive_gpio;
		dram_rtt_gpio = pdata->dram_rtt_gpio;
	}

	denali_base = ioremap_nocache(DMW_MPMC_BASE, SZ_4K);
	if (!denali_base) {
		dev_err(&pdev->dev, "could not map Denali controller!\n");
		return -ENOMEM;
	}

	cpu_clk = clk_get(&pdev->dev, "cpu");
	if (IS_ERR(cpu_clk)) {
		dev_err(&pdev->dev, "cpu clock missing!\n");
		ret = PTR_ERR(cpu_clk);
		goto free_denali;
	}

	pll3_clk = clk_get(&pdev->dev, "pll3");
	if (IS_ERR(pll3_clk)) {
		dev_err(&pdev->dev, "dram clock missing!\n");
		ret = PTR_ERR(pll3_clk);
		goto put_cpu_clk;
	}
	dram_freq = clk_get_rate(pll3_clk);
	if (dram_freq <= 0) {
		dev_err(&pdev->dev, "unknown dram freq!\n");
		ret = -ENODEV;
		goto put_pll3_clk;
	}

	sysclk_clk = clk_get(&pdev->dev, "sysclk");
	if (IS_ERR(sysclk_clk)) {
		dev_err(&pdev->dev, "sysclk clock missing!\n");
		ret = PTR_ERR(sysclk_clk);
		goto put_pll3_clk;
	}

	if (gpio_is_valid(overdrive_gpio)) {
		ret = gpio_request(overdrive_gpio, "pm-overdrive");
		if (ret) {
			dev_err(&pdev->dev, "Unable to get overdrive GPIO\n");
			goto put_sysclk_clk;
		}
	}

	if (gpio_is_valid(dram_rtt_gpio)) {
		ret = gpio_request(dram_rtt_gpio, "pm-dram-rtt");
		if (ret) {
			dev_err(&pdev->dev, "Unable to get dram-rtt GPIO\n");
			goto free_overdrive;
		}
	}

	/*
	 * Apply the input buffer switching irregardless of the chip revision
	 * because it saves some additional power also on RevB. Should be
	 * investigated by the system team...
	 */
	quirks = SLEEP_QUIRK_DIS_INP;

	/* determine RAM type */
	switch (denali_get(dram_type)) {
	case 0x0c:
		dmw_idle_dev = &dmw_idle_dev_lpddr2;
		set_lowpower_timings((2 + ns2clk(8, dram_freq)) * 3,
			(2 + ns2clk(140, dram_freq)) * 2);
		break;
	case 0x0f:
		/* DDR mode 4 is not stable on RevA due to unknown reasons. */
		if (dmw_get_chip_rev() == DMW_CHIP_DMW96_REV_A)
			dmw_idle_dev = &dmw_idle_dev_ddr2_rev_a;
		else
			dmw_idle_dev = &dmw_idle_dev_ddr2;
		set_lowpower_timings((3+2) * 3, (3+200) * 2);
		break;
	default:
		dev_err(&pdev->dev, "unknown RAM type!\n");
		ret = -ENODEV;
		goto free_dram_rtt;
	}

	/*
	 * Make sure we power down PLL1 in low power and run SYSCLK from 12MHz
	 */
	raw_local_irq_save(flags);
	local_fiq_disable();
	reg = cmu_readl(DMW_CMU_CPUCLKCNTRL);
	reg &= ~(3ul << 20); /* Run from CPUICLK in low power */
	reg |= 1ul << 19;    /* power down PLL1 in low power */
	cmu_writel(reg, DMW_CMU_CPUCLKCNTRL);
	raw_local_irq_restore(flags);

	suspend_set_ops(&dmw_pm_ops);

	ret = cpuidle_register_driver(&dmw_idle_driver);
	if (!ret)
		ret = cpuidle_register_device(dmw_idle_dev);

	if (ret)
		dev_warn(&pdev->dev, "idle driver not registered: %d\n", ret);

	return 0;

free_dram_rtt:
	if (gpio_is_valid(dram_rtt_gpio))
		gpio_free(dram_rtt_gpio);
free_overdrive:
	if (gpio_is_valid(overdrive_gpio))
		gpio_free(overdrive_gpio);
put_sysclk_clk:
	clk_put(sysclk_clk);
put_pll3_clk:
	clk_put(pll3_clk);
put_cpu_clk:
	clk_put(cpu_clk);
free_denali:
	iounmap(denali_base);
	return ret;
}

static struct platform_driver dmw_pm_plat_driver = {
	.driver = {
		.name	 = "dmw-pm",
		.owner	 = THIS_MODULE,
	},
};

static int __init dmw_pm_init(void)
{
	return platform_driver_probe(&dmw_pm_plat_driver, dmw_pm_probe);
}

late_initcall(dmw_pm_init);
