/*
 *  Keyboard driver for DSPG DMW models
 *
 *  Copyright (c) 2010 DSPG Technologies GmbH, Andreas Weissel
 *
 *  Based on corgikbd.c, which is based on xtkbd.c/locomkbd.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <linux/delay.h>
#include <linux/init.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/jiffies.h>
#include <linux/module.h>
#include <linux/slab.h>

#include <asm/io.h>
#include <mach/hardware.h>
#include <mach/keypad.h>

#define DRIVER_NAME "dmw-keypad"

#define DMW_KEYPAD_ACTIVATE_DELAY  20 /* us */
#define DMW_KEYPAD_DEBOUNCE_DELAY  50 /* ms */

/* Up to 8 rows and 8 columns are supported, so SCANCODE maps row and column
 * number to a value in the range of 1..64. */
#define SCANCODE(r,c) (((r)<<3) + (c) + 1)

struct dmw_keypad {
	struct input_dev *input;

	struct timer_list timer;
	int irq;
	int irq_disabled;
	void __iomem *regs;

	struct dmw_keypad_config *config;
};

static inline void
dmw_keypad_activate_col(struct dmw_keypad *dmw_keypad, int col)
{
	/* STROBE col -> High, not col -> HiZ */
	writel(1 << col, dmw_keypad->regs + DMW_KBD_COL);
}

static inline void
dmw_keypad_activate_all(struct dmw_keypad *dmw_keypad)
{
	/* STROBE ALL -> High */
	writel((1 << dmw_keypad->config->cols) - 1, dmw_keypad->regs + DMW_KBD_COL);
}

static int
dmw_keypad_key_pressed(struct dmw_keypad *dmw_keypad)
{
	return (readl(dmw_keypad->regs + DMW_KBD_ROW) &
	        ((1<<dmw_keypad->config->rows)-1));
}

static void
dmw_keypad_setup_sense_lines_and_strobe_high(struct dmw_keypad *dmw_keypad)
{
	unsigned long val;

	val =  ((1 << dmw_keypad->config->cols) - 1) |
	      (((1 << dmw_keypad->config->rows) - 1) << 8);
	writel(val, dmw_keypad->regs + DMW_KBD_CFG);
}

static void
dmw_keypad_setup_keys(struct dmw_keypad *dmw_keypad, struct input_dev *input_dev)
{
	int i, j;

	for (i = 0; i < dmw_keypad->config->rows; i++) {
		for (j = 0; j < dmw_keypad->config->cols; j++)
			set_bit(SCANCODE(i,j), input_dev->keybit);
	}
}

/*
 * The dw keyboard generates interrupts as long as a key is pressed.
 * When a key is pressed, we disable the interrupt and enable a timer to
 * periodically can the keyboard to detect when the key is released.
 */

/* Scan the hardware keyboard and push any changes up through the input layer */
static void
dmw_keypad_scankeyboard(unsigned long data)
{
	struct dmw_keypad *dmw_keypad = (struct dmw_keypad *)data;
	unsigned int col, row, row_mask;
	unsigned int num_pressed = 0;

	for (col = 0; col < dmw_keypad->config->cols; col++) {
		/*
		 * Discharge the output driver capacitance
		 * in the keyboard matrix. (Yes it is significant..)
		 */
		dmw_keypad_activate_col(dmw_keypad, col);
		udelay(DMW_KEYPAD_ACTIVATE_DELAY);

		row_mask = dmw_keypad_key_pressed(dmw_keypad);

		for (row = 0; row < dmw_keypad->config->rows; row++) {
			input_report_key(dmw_keypad->input, SCANCODE(row, col),
			                 !!(row_mask & (1 << row)));
			if (row_mask & (1 << row))
				num_pressed++;
		}
	}

	input_sync(dmw_keypad->input);

	dmw_keypad_activate_all(dmw_keypad);

	/* if any keys are pressed, enable the timer */
	if (num_pressed)
		mod_timer(&dmw_keypad->timer, jiffies +
		          msecs_to_jiffies(DMW_KEYPAD_DEBOUNCE_DELAY));
	else {
		/* we finished the polling phase, so register for the interrupt */
		udelay(DMW_KEYPAD_ACTIVATE_DELAY);
		dmw_keypad->irq_disabled = 0;
		enable_irq(dmw_keypad->irq);
	}
}

static irqreturn_t
dmw_keypad_interrupt(int irq, void *dev_id)
{
	struct dmw_keypad *dmw_keypad = (struct dmw_keypad *)dev_id;

	/* Scan the keyboard once. Further scans will be triggered by the
	 * timer. Keep the interrupt disabled in the meantime because the
	 * scanning will trigger it again. */
	disable_irq_nosync(dmw_keypad->irq);
	dmw_keypad->irq_disabled = 1;
	mod_timer(&dmw_keypad->timer, jiffies);

	return IRQ_HANDLED;
}

#ifdef CONFIG_PM
static int dmw_keypad_suspend(struct platform_device *dev, pm_message_t state)
{
	struct dmw_keypad *dmw_keypad = platform_get_drvdata(dev);

	if (device_may_wakeup(&dev->dev))
		enable_irq_wake(dmw_keypad->irq);

	/*
	 * Prevent a race with the interrupt / scanning function.
	 */
	disable_irq(dmw_keypad->irq);
	del_timer_sync(&dmw_keypad->timer);
	if (dmw_keypad->irq_disabled) {
		enable_irq(dmw_keypad->irq);
		dmw_keypad->irq_disabled = 0;
	}
	enable_irq(dmw_keypad->irq);

	return 0;
}
static int dmw_keypad_resume(struct platform_device *dev)
{
	struct dmw_keypad *dmw_keypad = platform_get_drvdata(dev);

	if (device_may_wakeup(&dev->dev))
		disable_irq_wake(dmw_keypad->irq);

	return 0;
}
#else
#define dmw_keypad_suspend NULL
#define dmw_keypad_resume  NULL
#endif

static int
dmw_keypad_probe(struct platform_device *pdev)
{
	struct dmw_keypad *dmw_keypad;
	struct input_dev *input_dev;
	struct resource *res;
	int ret;

	dmw_keypad = kzalloc(sizeof(struct dmw_keypad), GFP_KERNEL);
	if (!dmw_keypad)
		return -ENOMEM;

	input_dev = input_allocate_device();
	if (!input_dev) {
		kfree(dmw_keypad);
		return -ENOMEM;
	}

	platform_set_drvdata(pdev, dmw_keypad);
	dmw_keypad->config = (struct dmw_keypad_config *)pdev->dev.platform_data;
	dmw_keypad->input = input_dev;

	dmw_keypad->irq = platform_get_irq(pdev, 0);
	if (dmw_keypad->irq < 0) {
		dev_err(&pdev->dev, "no IRQ resource defined\n");
		ret = -ENODEV;
		goto out_kfree;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (res == NULL) {
		dev_err(&pdev->dev, "no memory resource defined\n");
		ret = -ENODEV;
		goto out_kfree;
	}

	/* res = request_mem_region(res->start, res->end - res->start +1,
	                            pdev->name); */

	dmw_keypad->regs = ioremap(res->start, res->end - res->start + 1);
	if (dmw_keypad->regs == NULL) {
		dev_err(&pdev->dev, "failed to ioremap() registers\n");
		ret = -ENODEV;
		goto out_kfree;
	}

	/* Init Keyboard rescan timer */
	init_timer(&dmw_keypad->timer);
	dmw_keypad->timer.function = dmw_keypad_scankeyboard;
	dmw_keypad->timer.data = (unsigned long)dmw_keypad;

	input_dev->name = "DMW Keyboard";
	input_dev->phys = "dmw_keypad/input0";
	input_dev->id.bustype = BUS_HOST;
	input_dev->id.vendor = 0x0001;
	input_dev->id.product = 0x0001;
	input_dev->id.version = 0x0100;
	input_dev->dev.parent = &pdev->dev;
	input_set_drvdata(input_dev, dmw_keypad);

	input_dev->evbit[0] = BIT(EV_KEY);
	input_dev->keycodesize = sizeof(unsigned char);

	dmw_keypad_setup_keys(dmw_keypad, input_dev);

	clear_bit(0, input_dev->keybit);

	ret = input_register_device(dmw_keypad->input);
	if (ret < 0) {
		dev_err(&pdev->dev, "failed to register input device\n");
		goto out_kfree;
	}

	/* Set Strobe lines as outputs - set high */
	dmw_keypad_setup_sense_lines_and_strobe_high(dmw_keypad);
	dmw_keypad_activate_all(dmw_keypad);

	/* Setup sense interrupts - RisingEdge Detect, sense lines as inputs */
	ret = request_irq(dmw_keypad->irq, dmw_keypad_interrupt,
	                  IRQF_TRIGGER_RISING, "dmw_keypad", dmw_keypad);
	if (ret < 0) {
		dev_err(&pdev->dev, "cannot get irq\n");
		goto out_irq;
	}

	/* Keypad is wakeup capable by default */
	device_init_wakeup(&pdev->dev, 1);

	return 0;

out_irq:
	input_unregister_device(dmw_keypad->input);
	dmw_keypad->input = NULL;
out_kfree:
	input_free_device(dmw_keypad->input);
	kfree(dmw_keypad);

	return ret;
}

static int
dmw_keypad_remove(struct platform_device *pdev)
{
	struct dmw_keypad *dmw_keypad = platform_get_drvdata(pdev);

	free_irq(dmw_keypad->irq, dmw_keypad);

	del_timer_sync(&dmw_keypad->timer);

	input_unregister_device(dmw_keypad->input);

	kfree(dmw_keypad);

	return 0;
}

static struct platform_driver dmw_keypad_driver = {
	.probe		= dmw_keypad_probe,
	.remove		= dmw_keypad_remove,
	.suspend	= dmw_keypad_suspend,
	.resume		= dmw_keypad_resume,
	.driver		= {
		.name	= DRIVER_NAME,
	},
};

static int __init dmw_keypad_init(void)
{
	return platform_driver_register(&dmw_keypad_driver);
}

static void __exit dmw_keypad_exit(void)
{
	platform_driver_unregister(&dmw_keypad_driver);
}

module_init(dmw_keypad_init);
module_exit(dmw_keypad_exit);

MODULE_AUTHOR("Andreas Weissel <andreas.weissel@dspg.com>");
MODULE_DESCRIPTION("DMW Keypad Driver");
MODULE_LICENSE("GPL");
