/*
 * DMA memcpy test case for GDMAC driver.
 *
 * Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/dma-mapping.h>
#include <linux/dmw96dma.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/wait.h>

static void fill_buffer(u32 *buf, size_t length, int i)
{
	length >>= 2;

	while (length--)
		*buf++ = i++;
}

static int cmp_buffer(u32 *buf, size_t length, int i)
{
	length >>= 2;

	while (length--)
		if (*buf++ != i++)
			return 1;

	return 0;
}

static volatile int done;

struct memcpy_ctx {
	struct dmw96dma_list *list;
	void *src;
	void *dst;
	size_t length;
	char *msg;
	wait_queue_head_t *wq;
};

static void memcpy_finish(int status, void *context)
{
	struct memcpy_ctx *ctx = (struct memcpy_ctx *)context;

	if (status)
		printk("DMA memcpy failed: %d\n", status);

	/* Very bad style to do this in tasklet context but its just a test... */
	if (cmp_buffer(ctx->src, ctx->length, 42))
		printk(" src#%s", ctx->msg);
	if (cmp_buffer(ctx->src+ctx->length, 32, 0xdead))
		printk(" end#%s", ctx->msg);
	if (cmp_buffer(ctx->dst, ctx->length, 42))
		printk(" dst#%s", ctx->msg);
	if (cmp_buffer(ctx->dst+ctx->length, 32, 0xf00d))
		printk(" ovr#%s", ctx->msg);

	if (ctx->wq) {
		done = 1;
		wake_up(ctx->wq);
	}

	printk(ctx->msg);

	dmw96dma_free_list(ctx->list);
	kfree(ctx->src);
	kfree(ctx->dst);
	kfree(ctx);
}

static void test_memcpy(size_t length, int wait, char *msg)
{
	struct memcpy_ctx *ctx;
	DECLARE_WAIT_QUEUE_HEAD_ONSTACK(wq);
	int ret;

	ctx = kzalloc(sizeof(*ctx), GFP_KERNEL);
	if (!ctx)
		return;
	ctx->src = kmalloc(length+32, GFP_KERNEL);
	if (!ctx->src)
		goto free1;
	ctx->dst = kmalloc(length+32, GFP_KERNEL);
	if (!ctx->dst)
		goto free2;

	ctx->length = length;
	ctx->msg = msg;
	fill_buffer(ctx->src, length, 42);
	fill_buffer(ctx->src+length, 32, 0xdead);
	fill_buffer(ctx->dst, length, 1337);
	fill_buffer(ctx->dst+length, 32, 0xf00d);

	ctx->list = dmw96dma_alloc_memcpy_list(memcpy_finish, ctx);
	if (!ctx->list)
		goto free3;

	ret = dmw96dma_add_memcpy_buf(ctx->list, ctx->src, ctx->dst, length, NULL, NULL);
	if (ret)
		goto free4;

	if (wait) {
		ctx->wq = &wq;
		done = 0;
	}

	ret = dmw96dma_submit(ctx->list);
	if (ret)
		goto free4;

	if (wait) {
		wait_event_timeout(wq, done, HZ);
		if (!done) {
			printk("DMA emcpy timed out!\n");
			goto free4;
		}
	}

	return;

free4:
	dmw96dma_free_list(ctx->list);
free3:
	kfree(ctx->dst);
free2:
	kfree(ctx->src);
free1:
	kfree(ctx);
}

static int memcpy_test_init(void)
{
	printk("DMA memcpy test...");

	test_memcpy(16,          1, "1");
	test_memcpy(32,          1, "2");
	test_memcpy(17*4,        1, "3");
	test_memcpy(256,         1, "4");
	test_memcpy(512,         0, "5");
	test_memcpy(512+4,       0, "6");
	test_memcpy(16*1024+64,  0, "7");
	test_memcpy(16*1024+144, 1, "8");

	printk("...done\n");

	return 0;
}

late_initcall(memcpy_test_init);
