/*
 * drivers/power/dp52-battery.c - battery driver for the DP52 chip
 *
 * Copyright (C) 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <linux/bitops.h>
#include <linux/ctype.h>
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/reboot.h>
#include <linux/regulator/consumer.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/wakelock.h>

#include <linux/mfd/core.h>
#include <linux/mfd/dp52/core.h>
#include <linux/mfd/dp52/battery.h>

/*
 * Charging does not start if the battery capacity is equal or above the
 * following percentage.
 */
#define CHRG_FULL_CAPACITY 93
#define CHRG_EMPTY_CAPACITY 5

#define MEASURE_VBAT	0
#define MEASURE_TEMP	1

/* When measuring the battery, we are actually measuring (battery + charging_voltage) */
/* When charing, charging_voltage is positive and being measured so we can substruct it */
/* When discharging, charging_voltage is negative, so we can't measure it, so we */
/* estimate it as 35mV */
#define DISCHARGE_OFFSET_MV		(-35)

/* How often to poll the battery status and update user-space */
#define POLL_INTERVAL			(60*HZ)

/* Minimum time between two measurements in ms */
#define DELAY_TIME			(10000)

static char *spec = "";
module_param(spec, charp, 0444);
MODULE_PARM_DESC(spec, "Battery specification");

enum state {
	CHRG_STATE_UNMOUNTED,
	CHRG_STATE_DISCHARGE,
	CHRG_STATE_MONITOR,
	CHRG_STATE_CHARGE,
	CHRG_STATE_THERMAL,
};

struct dp52_bat {
	struct device *dev;
	struct dp52 *dp52;
	struct dp52_bat_pdata *pdata;
	int cmp_irq;
	int cmp_irq_mask;

	enum state state;
	int present;
	struct workqueue_struct *wq;
	struct work_struct state_work;
	struct timer_list state_work_timer;
	struct work_struct measure_work;
	struct timer_list poll_timer;
	struct work_struct poll_work;
	wait_queue_head_t measure_wq;
	volatile unsigned long measure_cond;
	struct regulator *vcca;
	unsigned long update_time; /* jiffies when last measurement */

	/* cached values for user space */
	int vbat;
	int temp;

	/* battery parameters */
	int vmin;		/* mV */
	int vmax;		/* mV */
	int temp_min_charge;	/* C */
	int temp_max_charge;	/* C */
	int temp_min_discharge;	/* C */
	int temp_max_discharge;	/* C */
	int chrg_voltage;	/* mV */
	int chrg_current;	/* mA */
	int chrg_cutoff_curr;	/* mA */
	int chrg_cutoff_time;	/* minutes */
	int ntc_r;		/* Ohm */
	int ntc_b;		/* dimensionless */
	int ntc_t;		/* C */
	int v1;			/* mV */
	int c1;			/* percent */
	int v2;			/* mV */
	int c2;			/* percent */

	struct power_supply psy;
	struct wake_lock wake_lock;
};

static enum power_supply_property dp52_bat_props[] = {
	POWER_SUPPLY_PROP_CAPACITY,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_TEMP,
	POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN,
	POWER_SUPPLY_PROP_VOLTAGE_MIN_DESIGN,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
};

#define Q_SHIFT 12
#define Q_HALF (1 << (Q_SHIFT-1))
#define Q(x) ((x) << Q_SHIFT)

static int q_mul(int a, int b)
{
	return (a * b + Q_HALF) >> Q_SHIFT;
}

static int q_div(int a, int b)
{
	int x = (a << Q_SHIFT) + b / 2;
	return x / b;
}

/*
 * Returns the natural logarithm of a fixed point number.
 *
 * The function approximates the logarithm with the Taylor series which
 * converges for 0 < x < 2. Outside of this range we're using the fact that:
 *
 *   ln(1/x) = -ln(x)
 */
static int ln(int x)
{
	int k, l, xx;

	if (x >= Q(2)) {
		return -ln(q_div(Q(1), x));
	} else {
		x = x - Q(1);
	}


	l = x;
	xx = x;
	for (k=2; k<8; k+=2) {
		xx = q_mul(xx, x);
		l -= q_div(xx, Q(k));
		xx = q_mul(xx, x);
		l += q_div(xx, Q(k + 1));
	}

	return l;
}

static int calculate_bat_capacity(struct dp52_bat *bat);

static void poll_timeout(unsigned long priv)
{
	struct dp52_bat *bat = (struct dp52_bat *)priv;

	schedule_work(&bat->poll_work);
}

static void poll_work(struct work_struct *work)
{
	struct dp52_bat *bat = container_of(work, struct dp52_bat, poll_work);
	/* update user-space */
	power_supply_changed(&bat->psy);
	mod_timer(&bat->poll_timer, jiffies + POLL_INTERVAL);
}

static int dp52_bat_measure(struct dp52_bat *bat, int property)
{
	int ret;

	set_bit(property, &bat->measure_cond);
	queue_work(bat->wq, &bat->measure_work);

	ret = wait_event_interruptible(bat->measure_wq, bat->measure_cond == 0);
	if (ret)
		return ret;

	return bat->present ? 0 : -ENODEV;
}

static int dp52_bat_get_property(struct power_supply *psy,
                                 enum power_supply_property psp,
                                 union power_supply_propval *val)
{
	struct dp52_bat *bat = container_of(psy, struct dp52_bat, psy);
	int ret;

	switch (psp) {
		case POWER_SUPPLY_PROP_PRESENT:
			val->intval = bat->state != CHRG_STATE_UNMOUNTED;
			break;
		case POWER_SUPPLY_PROP_STATUS:
			switch (bat->state) {
				case CHRG_STATE_UNMOUNTED:
					val->intval = POWER_SUPPLY_STATUS_UNKNOWN;
					break;
				case CHRG_STATE_DISCHARGE:
					val->intval = POWER_SUPPLY_STATUS_DISCHARGING;
					break;
				case CHRG_STATE_MONITOR:
					val->intval = POWER_SUPPLY_STATUS_DISCHARGING;
					break;
				case CHRG_STATE_CHARGE:
					val->intval = POWER_SUPPLY_STATUS_CHARGING;
					break;
				case CHRG_STATE_THERMAL:
					val->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
					break;
			}
			break;
		case POWER_SUPPLY_PROP_TECHNOLOGY:
			val->intval = POWER_SUPPLY_TECHNOLOGY_LIPO;
			break;
		case POWER_SUPPLY_PROP_VOLTAGE_NOW:
			if ((ret = dp52_bat_measure(bat, MEASURE_VBAT)))
				return ret;
			val->intval = bat->vbat * 1000; /* uV */
			break;
		case POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN:
			val->intval = bat->vmax * 1000; /* uV */
			break;
		case POWER_SUPPLY_PROP_VOLTAGE_MIN_DESIGN:
			val->intval = bat->vmin * 1000; /* V */
			break;
		case POWER_SUPPLY_PROP_TEMP:
			if ((ret = dp52_bat_measure(bat, MEASURE_TEMP)))
				return ret;
			val->intval = bat->temp * 10; /* tenths of degree Celsius */
			break;
		case POWER_SUPPLY_PROP_CAPACITY:
			if ((ret = dp52_bat_measure(bat, MEASURE_VBAT)))
				return ret;

			val->intval = calculate_bat_capacity(bat);
			break;
		case POWER_SUPPLY_PROP_HEALTH:
			switch (bat->state) {
				case CHRG_STATE_THERMAL:
					if (bat->temp > 0)
						val->intval = POWER_SUPPLY_HEALTH_OVERHEAT;
					else
						val->intval = POWER_SUPPLY_HEALTH_COLD;
					break;
				case CHRG_STATE_UNMOUNTED:
					val->intval = POWER_SUPPLY_HEALTH_UNKNOWN;
					break;
				default:
					val->intval = POWER_SUPPLY_HEALTH_GOOD;
			}
			break;
		default:
			return -EINVAL;
	}

	return 0;
}

static void dp52_bat_external_power_changed(struct power_supply *psy)
{
	struct dp52_bat *bat = container_of(psy, struct dp52_bat, psy);

	queue_work(bat->wq, &bat->state_work);
}

static void work_timeout(unsigned long priv)
{
	struct dp52_bat *bat = (struct dp52_bat *)priv;

	queue_work(bat->wq, &bat->state_work);
}

static void schedule_check(struct dp52_bat *bat)
{
	int interval;

	switch (bat->state) {
		case CHRG_STATE_CHARGE:
			interval = 30*HZ;
			break;
		case CHRG_STATE_THERMAL:
			interval = 60*HZ;
			break;
		default:
			WARN_ONCE(1, "%s(): invalid state %d\n", __func__, bat->state);
			return;
	}

	mod_timer(&bat->state_work_timer, jiffies + interval);
}

#define TRIG_ABOVE	0
#define TRIG_BELOW	1

static int set_vbat_trigger(struct dp52_bat *bat, int vbat, int round, int pol)
{
	int ret, sense, gain, attn;

	sense = vbat * bat->pdata->vbat_r2 / (bat->pdata->vbat_r1 + bat->pdata->vbat_r2);

	ret = dp52_auxcmp_calculate(bat->dp52, sense, round, &gain, &attn);
	if (ret < 0) {
		dev_err(bat->dev, "cannot calculate gain and attn!\n");
		return ret;
	}

	ret = dp52_auxcmp_enable(bat->dp52, bat->pdata->vbat_dcin, gain, attn);
	if (ret < 0) {
		dev_err(bat->dev, "cannot set voltage monitor at %dmV (%dmV)!\n",
		        vbat, sense);
		return ret;
	}

	dev_dbg(bat->dev, "start monitoring vbat to go %s %dmV (%d)\n",
	        pol ? "below" : "above", vbat, ret);

	if (pol)
		dp52_set_bits(bat->dp52, DP52_AUX_INTPOL, 1 << bat->pdata->vbat_dcin);
	else
		dp52_clr_bits(bat->dp52, DP52_AUX_INTPOL, 1 << bat->pdata->vbat_dcin);

	dp52_set_bits(bat->dp52, DP52_AUX_INTMSKN, 1 << bat->pdata->vbat_dcin);

	return 0;
}

static void clr_vbat_trigger(struct dp52_bat *bat)
{
	dp52_clr_bits(bat->dp52, DP52_AUX_INTMSKN, 1 << bat->pdata->vbat_dcin);
	dp52_auxcmp_disable(bat->dp52, bat->pdata->vbat_dcin);
}

static void enter_unmounted_state(struct dp52_bat *bat)
{
	int ret, gain, attn;

	regulator_enable(bat->vcca);
	bat->state = CHRG_STATE_UNMOUNTED;

	/* No temp sensor -> no battery sense */
	if (bat->pdata->temp_dcin < 0)
		return;

	ret = dp52_auxcmp_calculate(bat->dp52, 2000, DP52_AUXCMP_ROUND_DOWN, &gain, &attn);
	if (ret < 0) {
		dev_err(bat->dev, "enter_unmounted_state: cannot calculate gain and attn!\n");
		return;
	}

	/* Try to detect battery */
	ret = dp52_auxcmp_enable(bat->dp52, bat->pdata->temp_dcin, gain, attn);
	if (ret < 0) {
		dev_err(bat->dev, "cannot set temp trigger!\n");
		return;
	}

	dev_dbg(bat->dev, "start sensing battery, monitoring temp to go below %dmV\n", ret);

	dp52_set_bits(bat->dp52, DP52_AUX_INTPOL, 1 << bat->pdata->temp_dcin);
	dp52_set_bits(bat->dp52, DP52_AUX_INTMSKN, 1 << bat->pdata->temp_dcin);
}

static void exit_unmounted_state(struct dp52_bat *bat)
{
	if (bat->pdata->temp_dcin < 0)
		return;

	dev_dbg(bat->dev, "leave sensing\n");
	dp52_clr_bits(bat->dp52, DP52_AUX_INTMSKN, 1 << bat->pdata->temp_dcin);
	dp52_auxcmp_disable(bat->dp52, bat->pdata->temp_dcin);

	regulator_disable(bat->vcca);
}

static int enter_discharge_state(struct dp52_bat *bat)
{
	dev_dbg(bat->dev, "start discharging battery\n");

	bat->state = CHRG_STATE_DISCHARGE;

	return 0;
}

static void exit_discharging_state(struct dp52_bat *bat)
{
	dev_dbg(bat->dev, "leave discharging\n");
}

static int charge_start(struct dp52_bat *bat, int thermal)
{
	int vcurrent = thermal
		? 0 /* as small as it can get... */
		: (bat->pdata->ichg_r * bat->chrg_current / 1000);
	int vmax = bat->chrg_voltage * bat->pdata->vbat_r2 /
		(bat->pdata->vbat_r1 + bat->pdata->vbat_r2);
	int ret;
	int gain;
	int attn;

	regulator_enable(bat->vcca);

	ret = dp52_auxcmp_calculate(bat->dp52, vcurrent, DP52_AUXCMP_ROUND_UP, &gain, &attn);
	if (ret < 0) {
		dev_err(bat->dev, "cannot calculate gain and attn!\n");
		return ret;
	}

	ret = dp52_auxcmp_enable(bat->dp52, bat->pdata->ichg_dcin, gain, attn);
	if (ret < 0) {
		dev_err(bat->dev, "cannot set current trigger to %dmV!\n",
			vcurrent);
		return ret;
	} else
		vcurrent = ret;

	/* enable vbat_dcin comperator with pre-calibrated values, which gives EXACT vmax voltage */
	ret = dp52_auxcmp_get_calibrated_values(bat->dp52, bat->pdata->vbat_dcin, &gain, &attn);
	if (ret < 0) {
		dev_err(bat->dev, "cannot get calibration values\n");
		dp52_auxcmp_disable(bat->dp52, bat->pdata->ichg_dcin);
		return ret;
	}

	ret = dp52_auxcmp_enable(bat->dp52, bat->pdata->vbat_dcin, gain, attn);
	if (ret < 0) {
		dev_err(bat->dev, "cannot set voltage trigger to %dmV!\n", vmax);
		dp52_auxcmp_disable(bat->dp52, bat->pdata->ichg_dcin);
		return ret;
	}

	dev_dbg(bat->dev, "start charging vcurrent=%d, vmax=%d, thermal=%d\n",
		vcurrent, vmax, thermal);

	dp52_set_bits(bat->dp52, DP52_PWM0_CFG2, 1 << 13); /* enable PWM0 */
	/* enable timer if we're really charging */
	if (!thermal) {
		dp52_clr_bits(bat->dp52, DP52_PWM0_CRGTMR_CFG, 1 << 4);	/* clear timer done indication */
		dp52_set_bits(bat->dp52, DP52_PWM0_CRGTMR_CFG, 1 << 7);
	}

	/*
	 * Prevent the system from suspending. We have to monitor the battery
	 * conditions regularily so we cannot afford to sleep.
	 */
	wake_lock(&bat->wake_lock);

	if (thermal)
		bat->state = CHRG_STATE_THERMAL;
	else
		bat->state = CHRG_STATE_CHARGE;

	return 0;
}

static int charge_stop(struct dp52_bat *bat)
{
	dev_dbg(bat->dev, "stop charging\n");

	wake_unlock(&bat->wake_lock);

	dp52_clr_bits(bat->dp52, DP52_PWM0_CFG2, 1 << 13); /* disable PWM0 */
	dp52_clr_bits(bat->dp52, DP52_PWM0_CRGTMR_CFG, 1 << 7); /* disable timer */
	dp52_auxcmp_disable(bat->dp52, bat->pdata->vbat_dcin);
	dp52_auxcmp_disable(bat->dp52, bat->pdata->ichg_dcin);

	regulator_disable(bat->vcca);
	return 0;
}

static int enter_monitor_state(struct dp52_bat *bat)
{
	int vbat, ret;

	regulator_enable(bat->vcca);

	vbat = ((bat->vmax - bat->vmin) * CHRG_FULL_CAPACITY / 100) + bat->vmin;
	ret = set_vbat_trigger(bat, vbat, DP52_AUXCMP_ROUND_DOWN, TRIG_BELOW);
	if (ret)
		return ret;

	dev_dbg(bat->dev, "start monitoring battery\n");

	bat->state = CHRG_STATE_MONITOR;

	return 0;
}

static int monitor_stop(struct dp52_bat *bat)
{
	dev_dbg(bat->dev, "stop monitoring\n");
	clr_vbat_trigger(bat);

	regulator_disable(bat->vcca);
	return 0;
}

static int __charger_connected(struct device *dev, void *data)
{
	union power_supply_propval ret = {0,};
	struct power_supply *psy = (struct power_supply *)data;
	struct power_supply *epsy = dev_get_drvdata(dev);
	int i;

	for (i = 0; i < epsy->num_supplicants; i++) {
		if (!strcmp(epsy->supplied_to[i], psy->name)) {
			if (epsy->get_property(epsy,
				  POWER_SUPPLY_PROP_ONLINE, &ret))
				continue;
			if (!ret.intval)
				return 0;

			/* Query max power. If not available assume 2A. */
			if (epsy->get_property(epsy,
				  POWER_SUPPLY_PROP_CURRENT_MAX, &ret))
				return 2000;
			else
				return ret.intval;
		}
	}

	return 0;
}

static int charger_connected(struct dp52_bat *bat)
{
	int max_curr;

	max_curr = class_for_each_device(power_supply_class, NULL, &bat->psy,
					 __charger_connected);

	return max_curr >= bat->chrg_current;
}

static void update_vbat(struct dp52_bat *bat)
{
	int charging;
	int offset;

	regulator_enable(bat->vcca);

	charging = bat->state == CHRG_STATE_CHARGE ||
		bat->state == CHRG_STATE_THERMAL;

	if (charging) {
		/* stop charging and wait for voltage to stabilize */
		dp52_clr_bits(bat->dp52, DP52_PWM0_CFG2, 1 << 13);
		msleep(100);
	}

	offset = DISCHARGE_OFFSET_MV;

	bat->vbat = dp52_auxadc_measure(bat->dp52, bat->pdata->vbat_dcin)*
		(bat->pdata->vbat_r1 + bat->pdata->vbat_r2) /
		bat->pdata->vbat_r2;

	bat->vbat -= offset;

	if (charging) {
		/* re-enable charging */
		dp52_set_bits(bat->dp52, DP52_PWM0_CFG2, 1 << 13);
	}

	dev_dbg(bat->dev, "vbat: %dmV (offset: %dmV)\n", bat->vbat, offset);
	regulator_disable(bat->vcca);
}

static int calculate_bat_capacity(struct dp52_bat *bat)
{
	int capacity;

	/* the voltage<->capacity graph has 3 linear parts:
	 * 1) (bat->vmin,0)		- (bat->v1,bat->c1)
	 * 2) (bat->v1,bat->c1)	- (bat->v2,bat->c2)
	 * 3) (bat->v2,bat->c2)	- (bat->vmax, 100)
	 */
	int v[2];
	int c[2];

	if (bat->vbat < bat->v1) {
		/* 1 */
		capacity = 0;
		v[0] = bat->vmin;
		c[0] = 0;
		v[1] = bat->v1;
		c[1] = bat->c1;
	}
	else if (bat->vbat < bat->v2) {
		/* 2 */
		capacity = bat->c1;
		v[0] = bat->v1;
		c[0] = bat->c1;
		v[1] = bat->v2;
		c[1] = bat->c2;
	}
	else {		/* bat->vbat >= bat->v2 */
		/* 3 */
		capacity = bat->c2;
		v[0] = bat->v2;
		c[0] = bat->c2;
		v[1] = bat->vmax;
		c[1] = 100;
	}
	capacity += (bat->vbat - v[0]) * (c[1] - c[0]) / (v[1] - v[0]);

	/* in case the voltage is above bat->vmax, we might get more than 100%... */
	if (capacity > 100)
		capacity = 100;

	dev_dbg(bat->dev, "raw capacity: %d%%\n", capacity);

	return capacity;
}

static int get_batt_capacity(struct dp52_bat *bat)
{
	update_vbat(bat);
	return calculate_bat_capacity(bat);
}

static void update_temp(struct dp52_bat *bat)
{
	int vtemp, rtemp, vcca, offset, charging;

	regulator_enable(bat->vcca);

	if (bat->pdata->temp_dcin < 0) {
		/* Fake temp */
		bat->temp = 25;
		goto done;
	}

	vcca = regulator_get_voltage(bat->vcca);
	WARN_ONCE(vcca <= 0, "Cannot get VCCA voltage: %d!\n", vcca);
	if (vcca <= 0) {
		/* Fake temp */
		bat->temp = 25;
		goto done;
	} else
		vcca /= 1000; /* convert to mV */

	vtemp = dp52_auxadc_measure(bat->dp52, bat->pdata->temp_dcin);

	if (vtemp < 50 || vtemp > 1950) {
		if (bat->present) {
			/* invalid temp -> battery not present */
			bat->present = 0;
			queue_work(bat->wq, &bat->state_work);
		}
		regulator_disable(bat->vcca);
		return;
	}

	charging = bat->state == CHRG_STATE_CHARGE ||
		bat->state == CHRG_STATE_THERMAL;

	if (charging) {
		/* subtract offset of shunt resistor */
		offset = dp52_auxadc_measure(bat->dp52, bat->pdata->ichg_dcin);
	}
	else
		offset = DISCHARGE_OFFSET_MV;

	rtemp = (vtemp - offset) * bat->pdata->temp_r / (vcca - offset - vtemp);
	bat->temp = bat->ntc_b * (bat->ntc_t+273) /
		(bat->ntc_b + ((ln(Q(rtemp) / bat->ntc_r) * (bat->ntc_t+273)) >> Q_SHIFT))
		- 273;

	dev_dbg(bat->dev, "temp: %dC (%dmV)\n", bat->temp, vtemp);

done:
	if (!bat->present) {
		bat->present = 1;
		queue_work(bat->wq, &bat->state_work);
	}
	
	regulator_disable(bat->vcca);
}

static int thermal_limit_reached(struct dp52_bat *bat)
{
	update_temp(bat);
	return bat->temp < bat->temp_min_charge || bat->temp > bat->temp_max_charge;
}

static int current_limit_reached(struct dp52_bat *bat)
{
	int voltage;
	int cur;

	voltage = dp52_auxadc_measure(bat->dp52, bat->pdata->ichg_dcin);
	cur = voltage * 1000 / bat->pdata->ichg_r;

	dev_dbg(bat->dev, "charge current: %dmA\n", cur);

	return cur < bat->chrg_cutoff_curr;
}

static int charge_timer_expired(struct dp52_bat *bat)
{
	return !!(dp52_read(bat->dp52, DP52_PWM0_CRGTMR_CFG) & 0x10);
}

static void state_do_unmounted(struct dp52_bat *bat)
{
	update_temp(bat); /* also does presence detection */

	if (bat->present) {
		exit_unmounted_state(bat);
		enter_discharge_state(bat);
		queue_work(bat->wq, &bat->state_work);
	}
}

static void state_do_discharge(struct dp52_bat *bat)
{
	if (!bat->present) {
		exit_discharging_state(bat);
		enter_unmounted_state(bat);
	} else if (charger_connected(bat)) {
		exit_discharging_state(bat);
		if (get_batt_capacity(bat) < CHRG_FULL_CAPACITY) {
			charge_start(bat, thermal_limit_reached(bat));
			schedule_check(bat);
		} else
			enter_monitor_state(bat);
	} else {
		regulator_enable(bat->vcca);
		set_vbat_trigger(bat, bat->vmin, DP52_AUXCMP_ROUND_UP, TRIG_BELOW); 

		if (!dp52_auxcmp_get(bat->dp52, bat->pdata->vbat_dcin)) {
			/*
			 * Battery below vmin! Either the battery was removed and we're
			 * running from a fixed power supply (e.g. on EVB) or the
			 * battery is really depleted. In this case we unconditionally
			 * power down the device.
			 */
			update_temp(bat);
			if (bat->present) {
				machine_power_off();
				dev_emerg(bat->dev, "Could not shut down! Battery may get damaged!");
			} else {
				exit_discharging_state(bat);
				enter_unmounted_state(bat);
			}
		}

		clr_vbat_trigger(bat);
		regulator_disable(bat->vcca);
	}
}

static void state_do_charge(struct dp52_bat *bat)
{
	if (!bat->present) {
		dev_dbg(bat->dev, "battery not present\n");
		charge_stop(bat);
		enter_unmounted_state(bat);
	} else if (!charger_connected(bat)) {
		dev_dbg(bat->dev, "charger not connected\n");
		charge_stop(bat);
		enter_discharge_state(bat);
	} else if (thermal_limit_reached(bat)) {
		dev_dbg(bat->dev, "thermal limit reached\n");
		charge_stop(bat);
		charge_start(bat, 1);
		schedule_check(bat);
	} else if (current_limit_reached(bat)) {
		dev_dbg(bat->dev, "current limit reached\n");
		charge_stop(bat);
		enter_monitor_state(bat);
	} else if (charge_timer_expired(bat)) {
		dev_dbg(bat->dev, "charge timer expired\n");
		charge_stop(bat);
		enter_monitor_state(bat);
	} else
		schedule_check(bat);
}

static void state_do_monitor(struct dp52_bat *bat)
{
	if (!bat->present) {
		monitor_stop(bat);
		enter_unmounted_state(bat);
	} else if (charger_connected(bat)) {
		if (!dp52_auxcmp_get(bat->dp52, bat->pdata->vbat_dcin)) {
			monitor_stop(bat);
			charge_start(bat, thermal_limit_reached(bat));
			schedule_check(bat);
		}
	} else {
		monitor_stop(bat);
		enter_discharge_state(bat);
	}
}

static void state_do_thermal(struct dp52_bat *bat)
{
	if (!bat->present) {
		charge_stop(bat);
		enter_unmounted_state(bat);
	} else if (charger_connected(bat)) {
		if (!thermal_limit_reached(bat)) {
			charge_stop(bat);
			charge_start(bat, 0);
		}
		schedule_check(bat);
	} else {
		charge_stop(bat);
		enter_discharge_state(bat);
	}
}

static void state_work(struct work_struct *work)
{
	struct dp52_bat *bat = container_of(work, struct dp52_bat, state_work);
	enum state old_state = bat->state;

	switch (bat->state) {
		case CHRG_STATE_UNMOUNTED:
			state_do_unmounted(bat);
			break;
		case CHRG_STATE_DISCHARGE:
			state_do_discharge(bat);
			break;
		case CHRG_STATE_CHARGE:
			state_do_charge(bat);
			break;
		case CHRG_STATE_MONITOR:
			state_do_monitor(bat);
			break;
		case CHRG_STATE_THERMAL:
			state_do_thermal(bat);
			break;
	}

	if (bat->state != old_state)
		power_supply_changed(&bat->psy);
}

static void measure_work(struct work_struct *work)
{
	struct dp52_bat *bat = container_of(work, struct dp52_bat, measure_work);
	int skip_measurement = 0;

	regulator_enable(bat->vcca);

	if (bat->update_time && time_before(jiffies, bat->update_time +
	                                    msecs_to_jiffies(DELAY_TIME)))
		skip_measurement = 1;
	bat->update_time = jiffies;

	if (test_bit(MEASURE_VBAT, &bat->measure_cond)) {
		if (!skip_measurement)
			update_vbat(bat);
		clear_bit(MEASURE_VBAT, &bat->measure_cond);
	}

	if (test_bit(MEASURE_TEMP, &bat->measure_cond)) {
		if (!skip_measurement)
			update_temp(bat);
		clear_bit(MEASURE_TEMP, &bat->measure_cond);
	}

	wake_up(&bat->measure_wq);
	regulator_disable(bat->vcca);
}

static irqreturn_t dp52_bat_cmp_irq(int irq, void *priv)
{
	struct dp52_bat *bat = priv;
	int pending;

	pending = dp52_read(bat->dp52, DP52_AUX_INTSTAT);

	if (pending & bat->cmp_irq_mask) {
		queue_work(bat->wq, &bat->state_work);
		dp52_write(bat->dp52, DP52_AUX_INTSTAT, ~bat->cmp_irq_mask);
		return IRQ_HANDLED;
	}

	return IRQ_NONE;
}

static int __devinit get_bat_param(struct device *dev, char *tag, int *val)
{
	char *pos = strstr(spec, tag);
	int len = strlen(tag);

	if (!pos) {
		dev_err(dev, "missing '%s' battery parameter!\n", tag);
		return -EINVAL;
	}

	if (pos[len] != ':' || (!isdigit(pos[len+1]) && pos[len+1] != '-')) {
		dev_err(dev, "invalid battery parameter '%s'\n", tag);
		return -EINVAL;
	}

	*val = simple_strtol(pos + len + 1, NULL, 10);
	return 0;
}

static int __devinit bat_load_params(struct dp52_bat *bat)
{
	struct device *dev = bat->dev;
	int ret;

	/* parse calibration data */
	if ((ret = get_bat_param(dev, "vmin", &bat->vmin)))
		return ret;
	if ((ret = get_bat_param(dev, "vmax", &bat->vmax)))
		return ret;
	if ((ret = get_bat_param(dev, "tminc", &bat->temp_min_charge)))
		return ret;
	if ((ret = get_bat_param(dev, "tmind", &bat->temp_min_discharge)))
		return ret;
	if ((ret = get_bat_param(dev, "tmaxc", &bat->temp_max_charge)))
		return ret;
	if ((ret = get_bat_param(dev, "tmaxd", &bat->temp_max_discharge)))
		return ret;
	if ((ret = get_bat_param(dev, "cv", &bat->chrg_voltage)))
		return ret;
	if ((ret = get_bat_param(dev, "ci", &bat->chrg_current)))
		return ret;
	if ((ret = get_bat_param(dev, "cci", &bat->chrg_cutoff_curr)))
		return ret;
	if ((ret = get_bat_param(dev, "cct", &bat->chrg_cutoff_time)))
		return ret;
	if ((ret = get_bat_param(dev, "ntcr", &bat->ntc_r)))
		return ret;
	if ((ret = get_bat_param(dev, "ntcb", &bat->ntc_b)))
		return ret;
	if ((ret = get_bat_param(dev, "ntct", &bat->ntc_t)))
		return ret;
	if ((ret = get_bat_param(dev, "v1", &bat->v1)))
		return ret;
	if ((ret = get_bat_param(dev, "c1", &bat->c1)))
		return ret;
	if ((ret = get_bat_param(dev, "v2", &bat->v2)))
		return ret;
	if ((ret = get_bat_param(dev, "c2", &bat->c2)))
		return ret;

	return 0;
}

static int __devinit dp52_bat_probe(struct platform_device *pdev)
{
	struct dp52 *dp52 = dev_get_drvdata(pdev->dev.parent);
	struct mfd_cell *cell = pdev->mfd_cell;
	struct dp52_bat *bat;
	int ret;

	if (!cell->mfd_data) {
		dev_err(&pdev->dev, "missing platform data!\n");
		return -EINVAL;
	}

	bat = kzalloc(sizeof(*bat), GFP_KERNEL);
	if (!bat)
		return -ENOMEM;

	device_init_wakeup(&pdev->dev, 1);

	platform_set_drvdata(pdev, bat);

	bat->dev = &pdev->dev;
	bat->dp52 = dp52;
	bat->pdata = cell->mfd_data;
	INIT_WORK(&bat->state_work, state_work);
	INIT_WORK(&bat->measure_work, measure_work);
	init_waitqueue_head(&bat->measure_wq);
	setup_timer(&bat->state_work_timer, work_timeout, (unsigned long)bat);
	wake_lock_init(&bat->wake_lock, WAKE_LOCK_SUSPEND, "dp52-battery");
	bat->cmp_irq_mask = (1 << bat->pdata->vbat_dcin) |
	                    (1 << bat->pdata->ichg_dcin);
	if (bat->pdata->temp_dcin >= 0)
		bat->cmp_irq_mask |= (1 << bat->pdata->temp_dcin);

	if ((bat->cmp_irq = platform_get_irq_byname(pdev, "auxcomp")) < 0) {
		ret = bat->cmp_irq;
		dev_err(&pdev->dev, "no AUXCOMP IRQ!\n");
		goto err_free;
	}

	if ((ret = bat_load_params(bat)))
		goto err_free;

	/* Create workqueue */
	if (!(bat->wq = create_singlethread_workqueue("main-battery"))) {
		ret = -ENOMEM;
		goto err_free;
	}

	/* Get VCCA regulator and enable it */
	bat->vcca = regulator_get(&pdev->dev, "vcca");
	if (IS_ERR(bat->vcca)) {
		dev_err(&pdev->dev, "no vcca regulator!\n");
		ret = PTR_ERR(bat->vcca);
		goto err_del_wq;
	}

	/*
	 * Setup PWM0:
	 * modulo = 256 (counting 0..255)
	 * duty_cycle = 0xfe = 254 (almost maximum DC)
	 *   - cut off if above configured Vbat and Ichg
	 */
	dp52_write(bat->dp52, DP52_PWM0_CFG2, 7 << 9);	/* modulo = 256 */
	dp52_write(bat->dp52, DP52_PWM0_CFG1, 0x00fe |
		(0x1100 << bat->pdata->ichg_dcin) |
		(0x1100 << bat->pdata->vbat_dcin));

	/*
	 * Init timer, counting every 64 seconds. Counting
	 * starts as soon Vbat rises above cutoff threshold.
	 */
	dp52_write(bat->dp52, DP52_PWM0_CRGTMR_CFG, 0x0000);
	dp52_write(bat->dp52, DP52_PWM0_CRG_TIMEOUT,
		bat->chrg_cutoff_time * 60 / 64);
	dp52_write(bat->dp52, DP52_PWM0_CRGTMR_CFG, 0x000b |
		(0x0100 << bat->pdata->vbat_dcin));

	/* register power supply */
	bat->psy.name			= "main-battery";
	bat->psy.type			= POWER_SUPPLY_TYPE_BATTERY;
	bat->psy.properties		= dp52_bat_props;
	bat->psy.num_properties		= ARRAY_SIZE(dp52_bat_props);
	bat->psy.get_property		= dp52_bat_get_property;
	bat->psy.external_power_changed	= dp52_bat_external_power_changed;
	bat->psy.use_for_apm		= 1;

	if ((ret = power_supply_register(&pdev->dev, &bat->psy))) {
		dev_err(&pdev->dev, "failed to register power supply\n");
		goto err_put_vcca;
	}

	/* At the beginning we're in sense state */
	enter_unmounted_state(bat);

	/* grab IRQ */
	ret = request_threaded_irq(bat->cmp_irq, NULL, dp52_bat_cmp_irq, IRQF_SHARED,
	                           "dp52-battery", bat);
	if (ret) {
		dev_err(&pdev->dev, "failed to request IRQ %d: %d\n", bat->cmp_irq, ret);
		goto err_unregister;
	}

	/* get the state machine started */
	queue_work(bat->wq, &bat->state_work);

	INIT_WORK(&bat->poll_work, poll_work);
	setup_timer(&bat->poll_timer, poll_timeout, (unsigned long)bat);
	mod_timer(&bat->poll_timer, jiffies + POLL_INTERVAL);

	dev_info(&pdev->dev, "initialized\n");
	return 0;

err_unregister:
	power_supply_unregister(&bat->psy);
	del_timer_sync(&bat->state_work_timer);
err_del_wq:
	flush_workqueue(bat->wq);
	destroy_workqueue(bat->wq);
err_put_vcca:
	regulator_put(bat->vcca);
err_free:
	wake_lock_destroy(&bat->wake_lock);
	kfree(bat);
	return ret;
}

static int __devexit dp52_bat_remove(struct platform_device *pdev)
{
	struct dp52_bat *bat = platform_get_drvdata(pdev);

	free_irq(bat->cmp_irq, bat);
	power_supply_unregister(&bat->psy);
	del_timer_sync(&bat->state_work_timer);
	del_timer_sync(&bat->poll_timer);
	flush_workqueue(bat->wq);
	destroy_workqueue(bat->wq);
	regulator_disable(bat->vcca);
	regulator_put(bat->vcca);
	kfree(bat);
	return 0;
}

static void dp52_bat_shutdown(struct platform_device *pdev)
{
	struct dp52_bat *bat = platform_get_drvdata(pdev);

	dp52_clr_bits(bat->dp52, DP52_AUX_INTMSKN, 1 << bat->pdata->vbat_dcin);
	if (bat->pdata->temp_dcin >= 0) {
		dp52_clr_bits(bat->dp52, DP52_AUX_INTMSKN, 1 << bat->pdata->temp_dcin);
		dp52_auxcmp_disable(bat->dp52, bat->pdata->temp_dcin);
	}
	dp52_clr_bits(bat->dp52, DP52_PWM0_CFG2, 1 << 13); /* disable PWM0 */
	dp52_clr_bits(bat->dp52, DP52_PWM0_CRGTMR_CFG, 1 << 7); /* disable timer */
	dp52_auxcmp_disable(bat->dp52, bat->pdata->vbat_dcin);
	dp52_auxcmp_disable(bat->dp52, bat->pdata->ichg_dcin);
}

static int dp52_bat_suspend(struct device *dev)
{
	struct dp52_bat *bat = platform_get_drvdata(to_platform_device(dev));

	if (bat->state == CHRG_STATE_CHARGE || bat->state == CHRG_STATE_THERMAL) {
		dev_warn(dev, "charging; cannot suspend!");
		return -EBUSY;
	}

	return regulator_disable(bat->vcca);
}

static int dp52_bat_resume(struct device *dev)
{
	struct dp52_bat *bat = platform_get_drvdata(to_platform_device(dev));

	return regulator_enable(bat->vcca);
}

static const struct dev_pm_ops dp_bat_driver_pm_ops = {
	.suspend = dp52_bat_suspend,
	.resume  = dp52_bat_resume,
};

static struct platform_driver dp_bat_driver = {
	.driver = {
		.name  = "dp52-battery",
		.owner = THIS_MODULE,
		.pm    = &dp_bat_driver_pm_ops,
	},
	.probe  = dp52_bat_probe,
	.remove = __devexit_p(dp52_bat_remove),
	.shutdown = dp52_bat_shutdown,
};

static int __init dp52_bat_init(void)
{
	return platform_driver_register(&dp_bat_driver);
}

static void __exit dp52_bat_exit(void)
{
	platform_driver_unregister(&dp_bat_driver);
}

module_init(dp52_bat_init);
module_exit(dp52_bat_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("DSP Group Inc.");
MODULE_DESCRIPTION("DP52 LiIo battery driver");
MODULE_ALIAS("platform:dp52-battery");

