/*
 * TODO:
 * 	- dynamically setup var and fix on probe
 * 	- switch backlight on/off
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/mm.h>
#include <linux/fb.h>
#include <linux/clk.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <asm/uaccess.h> /*needed for copy_form_user*/
#include <linux/dma-mapping.h> /*for ama allocate*/
#include <linux/interrupt.h>
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
#include <asm/gpio.h>
#endif
#ifdef CONFIG_MACH_VERSATILE_BROADTILE
#include <mach/hardware.h>
#endif
#include <linux/semaphore.h>
#include <linux/string.h>
#include <linux/sysfs.h>
#include <linux/device.h>
#include <linux/notifier.h>
#include <video/dmw96osdm_w.h>
#include "dmw96osdm.h"

MODULE_DESCRIPTION("DMW96 OSDM driver");
MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");

const char* plane_name_arr[] = {"dmw96_main_video","dmw96_sub_video","dmw96_main_osd","dmw96_sub_osd1","dmw96_sub_osd2"};
static int dmw96_osdm_planes_sizes[NUM_OF_IN_PLANES] = {0};
static struct dmw96osdm* g_osdm = NULL;
static struct notifier_block notifier;

extern void __iomem *osdm_regs;
extern int get_osdm_parameters_from_lcdc(struct fb_info* display,  int (*osdm_vsync_cb)(void));
extern int switch_osdm_and_lcdc_address(struct fb_info* display);
extern void lcdc_enable_vsync_int_once(void);

const char* get_plane_name(unsigned int plane)
{
	if (plane > NUM_OF_IN_PLANES)
		return NULL;

	return plane_name_arr[plane];
}

EXPORT_SYMBOL(get_plane_name);

int get_plane_num(struct dmw96osdm* osdm, struct fb_info *info)
{
	unsigned int i = 0;

	for (i = 0 ; i < NUM_OF_IN_PLANES ; i++) {
            if (&osdm->uncommitted_planes[i].info == info)
				return i;
	}

	return -1;
}

EXPORT_SYMBOL(get_plane_num);

/*
* init_display()
*/
static int init_display(void)
{
	unsigned long flags;
	int res;
	struct fb_info display;
    
	res = get_osdm_parameters_from_lcdc(&display, &start_mix);

	if (SUCCEEDED(res)) {
		/* copy selected (needed) fields into commited copy of DISPLAY_PLANE */
		spin_lock_irqsave(g_osdm->copy_lock, flags);

		g_osdm->planes[DISPLAY_PLANE].info.var.xres = display.var.xres;
		g_osdm->planes[DISPLAY_PLANE].info.var.yres = display.var.yres;
		g_osdm->planes[DISPLAY_PLANE].info.var.reserved[0] = display.var.reserved[0];
		g_osdm->planes[DISPLAY_PLANE].info.fix.smem_start = display.fix.smem_start;
		g_osdm->planes[DISPLAY_PLANE].commited = 1;

		spin_unlock_irqrestore(g_osdm->copy_lock, flags);
	}

	return res; 
}

static int
dmw96osdm_notifier_callback(struct notifier_block *p,
                            unsigned long event, void *data)
{
	if (event == FB_EVENT_MODE_CHANGE_ALL) {
		PDEBUG("OSDM display config\n");
		init_display();
		return NOTIFY_OK;
	}

	return NOTIFY_DONE;
}

static int pan_osdm(void)
{
	int res = SUCCESS;
	struct fb_info display;

	dmw96osdm_display_address_get(&display);
	dmw96osdm_display_dim_get(&display);

	PDEBUG("OSDM (%d) display.add=0x%lx display.xres=%d display.yers=%d\n", __LINE__ ,display.fix.smem_start, display.var.xres, display.var.yres);
	res = switch_osdm_and_lcdc_address(&display);
	PDEBUG("OSDM (%d) display.add=0x%lx display.xres=%d display.yers=%d\n", __LINE__ ,display.fix.smem_start, display.var.xres, display.var.yres);

	if (SUCCEEDED(res)) {
		PDEBUG("OSDM (%d) display.add=0x%lx display.xres=%d display.yers=%d\n", __LINE__ ,display.fix.smem_start, display.var.xres, display.var.yres);
		dmw96osdm_display_address_set(&display);
	}
	
	return res;
}

static int dmw96_print_plane(struct fb_info *info , unsigned int plane_num)
{
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm || plane_num > DISPLAY_PLANE)
		return -1;
	
	PDEBUG("%s OSDM Set background red = 0x%4x green = 0x%4x blue = 0x%4x\n",plane_name_arr[plane_num],
																		get_red_from_rgb888(osdm->planes[DISPLAY_PLANE].bg_color)
																		,get_green_from_rgb888(osdm->planes[DISPLAY_PLANE].bg_color)
																		, get_blue_from_rgb888(osdm->planes[DISPLAY_PLANE].bg_color));
    PDEBUG("%s OSDM Set background is : %s \n", plane_name_arr[plane_num], ((osdm->planes[DISPLAY_PLANE].bg_enb > 0) ? "Enable" : "Dissable"));
    PDEBUG("%s OSDM display dithering is %d\n", plane_name_arr[plane_num], osdm->planes[DISPLAY_PLANE].dithering );


	PDEBUG("%s Plane is: %s \n", plane_name_arr[plane_num],(osdm->planes[plane_num].enb > 0) ? "Enable" : "Dissable");
	PDEBUG("%s Plane is: %s commited\n", plane_name_arr[plane_num],(osdm->planes[plane_num].commited > 0) ? "successfully" : "not");
	PDEBUG("%s Plane physical address:0x%lx len:%d \n", plane_name_arr[plane_num], osdm->planes[plane_num].info.fix.smem_start , osdm->planes[plane_num].info.fix.smem_len);

	PDEBUG("%s Config is 0x%x \n", plane_name_arr[plane_num],osdm->planes[plane_num].config );
	PDEBUG("%s Chroma key is: %s \n", plane_name_arr[plane_num],(osdm->planes[plane_num].chroma_enb > 0) ? "Enable" : "Dissable");
	PDEBUG("%s Per pixel alpha is: %s \n", plane_name_arr[plane_num],(osdm->planes[plane_num].per_pixel_alpha_enb > 0) ? "Enable" : "Dissable");
	PDEBUG("%s Mask top=%d bottom=%d right=%d left=%d\n", plane_name_arr[plane_num],osdm->planes[plane_num].mask.top ,  osdm->planes[plane_num].mask.bottom , osdm->planes[plane_num].mask.right , osdm->planes[plane_num].mask.left );
	PDEBUG("%s Position is x:%d y:%d \n", plane_name_arr[plane_num], osdm->planes[plane_num].pos.xpos , osdm->planes[plane_num].pos.ypos);
	PDEBUG("%s Dimention is x:%d y:%d \n", plane_name_arr[plane_num], osdm->planes[plane_num].info.var.xres , osdm->planes[plane_num].info.var.yres);
	PDEBUG("%s Rotate:%d \n", plane_name_arr[plane_num], info->var.rotate);
	PDEBUG("%s Position of transparnt is x:%d y:%d \n",plane_name_arr[plane_num], osdm->planes[plane_num].t_dim.xres , osdm->planes[plane_num].t_dim.yres);
	PDEBUG("%s Dimention of transparency is x:%d y:%d \n",plane_name_arr[plane_num], osdm->planes[plane_num].t_pos.xpos , osdm->planes[plane_num].t_pos.ypos);
	PDEBUG("%s Transprency pixel is red:%d green:%d blue:%d alpha:%d\n",plane_name_arr[plane_num],
					    get_red_from_abgr8888(osdm->planes[plane_num].t_pixel_color),
					    get_green_from_abgr8888(osdm->planes[plane_num].t_pixel_color),
					    get_blue_from_abgr8888(osdm->planes[plane_num].t_pixel_color),
						get_alpha_from_abgr8888(osdm->planes[plane_num].t_pixel_color) );
	PDEBUG("%s Per plane alpha is %d\n", plane_name_arr[plane_num],osdm->planes[plane_num].per_plane_alpha );
    PDEBUG("%s Chroma red is %d\n",plane_name_arr[plane_num], get_red_from_rgb888(osdm->planes[plane_num].chroma_key));
	PDEBUG("%s Chroma green is %d\n",plane_name_arr[plane_num], get_green_from_rgb888(osdm->planes[plane_num].chroma_key));
	PDEBUG("%s Chroma blue is %d\n",plane_name_arr[plane_num], get_blue_from_rgb888(osdm->planes[plane_num].chroma_key));
	PDEBUG("%s byte order [0] is %d\n",plane_name_arr[plane_num], osdm->planes[plane_num].byte_order[0]);
	PDEBUG("%s byte order [1] is %d\n",plane_name_arr[plane_num], osdm->planes[plane_num].byte_order[1]);
	PDEBUG("%s byte order [2] is %d\n",plane_name_arr[plane_num], osdm->planes[plane_num].byte_order[2]);
	PDEBUG("%s Lut base is %d\n",plane_name_arr[plane_num], osdm->planes[plane_num].lut_base);

	if (plane_num == SUB_VIDEO_PLANE || plane_num == MAIN_VIDEO_PLANE) {
		PDEBUG("%s Color control brightness %d\n",plane_name_arr[plane_num], osdm->planes[plane_num].video_color_control.brightness);
		PDEBUG("%s Color control saturation %d\n",plane_name_arr[plane_num], osdm->planes[plane_num].video_color_control.saturation);
		PDEBUG("%s Color control hue %d\n",plane_name_arr[plane_num], osdm->planes[plane_num].video_color_control.hue);
		PDEBUG("%s Color control contrast %d\n",plane_name_arr[plane_num], osdm->planes[plane_num].video_color_control.contrast);
	
		PDEBUG("%s Limits luma max %d\n", plane_name_arr[plane_num], osdm->planes[plane_num].video_limits.luma_max);
		PDEBUG("%s Limits luma min %d\n", plane_name_arr[plane_num], osdm->planes[plane_num].video_limits.luma_min);
		PDEBUG("%s Limits chroma max %d\n", plane_name_arr[plane_num], osdm->planes[plane_num].video_limits.chroma_max);
		PDEBUG("%s Limits chroma min %d\n", plane_name_arr[plane_num], osdm->planes[plane_num].video_limits.chroma_min);
	}

	return 0;
}

static int dmw96osdm_bitfield_set(struct fb_var_screeninfo* var,const unsigned int config)
{
	if (!var || ( !(IS_RGB32(config)) && !(IS_RGB16(config)) ) )
		return -1;

	switch (config)
	{
		case OSDM_RGB32_8888_ARGB:
			var->red.length = 8;	
			var->red.offset = 16;	
			var->green.length = 8;	
			var->green.offset = 8;	
			var->blue.length = 8;
			var->blue.offset = 0;
			var->transp.length = 8;	
			var->transp.offset = 24;	
			break;
		case OSDM_RGB32_8888_ARBG:
			var->red.length = 8;	
			var->red.offset = 16;	
			var->green.length = 8;	
			var->green.offset = 0;	
			var->blue.length = 8;
			var->blue.offset = 8;
			var->transp.length = 8;	
			var->transp.offset = 24;
			break;
		case OSDM_RGB32_8888_AGRB:
			var->red.length = 8;	
			var->red.offset = 8;	
			var->green.length = 8;	
			var->green.offset = 16;	
			var->blue.length = 8;
			var->blue.offset = 0;
			var->transp.length = 8;	
			var->transp.offset = 24;
			break;
		case OSDM_RGB32_8888_AGBR:
			var->red.length = 8;	
			var->red.offset = 0;	
			var->green.length = 8;	
			var->green.offset = 16;	
			var->blue.length = 8;
			var->blue.offset = 8;
			var->transp.length = 8;	
			var->transp.offset = 24;
			break;
		case OSDM_RGB32_8888_ABRG:
			var->red.length = 8;	
			var->red.offset = 8;	
			var->green.length = 8;	
			var->green.offset = 0;	
			var->blue.length = 8;
			var->blue.offset = 16;
			var->transp.length = 8;	
			var->transp.offset = 24;
			break;
		case OSDM_RGB32_8888_ABGR:
			var->red.length = 8;	
			var->red.offset = 0;	
			var->green.length = 8;	
			var->green.offset = 8;	
			var->blue.length = 8;
			var->blue.offset = 16;
			var->transp.length = 8;	
			var->transp.offset = 24;
			break;
		case OSDM_RGB32_8888_RGBA:
			var->red.length = 8;	
			var->red.offset = 24;	
			var->green.length = 8;	
			var->green.offset = 16;	
			var->blue.length = 8;
			var->blue.offset = 8;
			var->transp.length = 8;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB32_8888_RBGA:
			var->red.length = 8;	
			var->red.offset = 24;	
			var->green.length = 8;	
			var->green.offset = 8;	
			var->blue.length = 8;
			var->blue.offset = 16;
			var->transp.length = 8;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB32_8888_GRBA:
			var->red.length = 8;	
			var->red.offset = 16;	
			var->green.length = 8;	
			var->green.offset = 24;	
			var->blue.length = 8;
			var->blue.offset = 8;
			var->transp.length = 8;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB32_8888_GBRA:
			var->red.length = 8;	
			var->red.offset = 8;	
			var->green.length = 8;	
			var->green.offset = 24;	
			var->blue.length = 8;
			var->blue.offset = 16;
			var->transp.length = 8;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB32_8888_BRGA:
			var->red.length = 8;	
			var->red.offset = 16;	
			var->green.length = 8;	
			var->green.offset = 8;	
			var->blue.length = 8;
			var->blue.offset = 24;
			var->transp.length = 8;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB32_8888_BGRA:
			var->red.length = 8;	
			var->red.offset = 8;	
			var->green.length = 8;	
			var->green.offset = 16;	
			var->blue.length = 8;
			var->blue.offset = 24;
			var->transp.length = 8;	
			var->transp.offset = 0;
			break;
	
		case OSDM_RGB16_4444_ARGB:
			var->red.length = 4;	
			var->red.offset = 8;	
			var->green.length = 4;	
			var->green.offset = 4;	
			var->blue.length = 4;
			var->blue.offset = 0;
			var->transp.length = 4;	
			var->transp.offset = 12;
			break;
		case OSDM_RGB16_4444_ARBG:
			var->red.length = 4;	
			var->red.offset = 8;	
			var->green.length = 4;	
			var->green.offset = 0;	
			var->blue.length = 4;
			var->blue.offset = 4;
			var->transp.length = 4;	
			var->transp.offset = 12;
			break;
		case OSDM_RGB16_4444_AGRB:
			var->red.length = 4;	
			var->red.offset = 4;	
			var->green.length = 4;	
			var->green.offset = 8;	
			var->blue.length = 4;
			var->blue.offset = 0;
			var->transp.length = 4;	
			var->transp.offset = 12;
			break;
		case OSDM_RGB16_4444_AGBR:
			var->red.length = 4;	
			var->red.offset = 0;	
			var->green.length = 4;	
			var->green.offset = 8;	
			var->blue.length = 4;
			var->blue.offset = 4;
			var->transp.length = 4;	
			var->transp.offset = 12;
			break;
		case OSDM_RGB16_4444_ABRG:
			var->red.length = 4;	
			var->red.offset = 4;	
			var->green.length = 4;	
			var->green.offset = 0;	
			var->blue.length = 4;
			var->blue.offset = 8;
			var->transp.length = 4;	
			var->transp.offset = 12;
			break;
		case OSDM_RGB16_4444_ABGR:
			var->red.length = 4;	
			var->red.offset = 0;	
			var->green.length = 4;	
			var->green.offset = 4;	
			var->blue.length = 4;
			var->blue.offset = 8;
			var->transp.length = 4;	
			var->transp.offset = 12;
			break;
		case OSDM_RGB16_4444_RGBA:
			var->red.length = 4;	
			var->red.offset = 12;	
			var->green.length = 4;	
			var->green.offset = 8;	
			var->blue.length = 4;
			var->blue.offset = 4;
			var->transp.length = 4;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB16_4444_RBGA:
			var->red.length = 4;	
			var->red.offset = 12;	
			var->green.length = 4;	
			var->green.offset = 4;	
			var->blue.length = 4;
			var->blue.offset = 8;
			var->transp.length = 4;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB16_4444_GRBA:
			var->red.length = 4;	
			var->red.offset = 8;	
			var->green.length = 4;	
			var->green.offset = 12;	
			var->blue.length = 4;
			var->blue.offset = 4;
			var->transp.length = 4;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB16_4444_GBRA:
			var->red.length = 4;	
			var->red.offset = 4;	
			var->green.length = 4;	
			var->green.offset = 12;	
			var->blue.length = 4;
			var->blue.offset = 8;
			var->transp.length = 4;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB16_4444_BRGA:
			var->red.length = 4;	
			var->red.offset = 8;	
			var->green.length = 4;	
			var->green.offset = 4;	
			var->blue.length = 4;
			var->blue.offset = 12;
			var->transp.length = 4;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB16_4444_BGRA:
			var->red.length = 4;	
			var->red.offset = 4;	
			var->green.length = 4;	
			var->green.offset = 8;	
			var->blue.length = 4;
			var->blue.offset = 12;
			var->transp.length = 4;	
			var->transp.offset = 0;
			break;
		case OSDM_RGB16_565_RGB:
			var->red.length = 5;	
			var->red.offset = 11;	
			var->green.length = 6;	
			var->green.offset = 5;	
			var->blue.length = 5;
			var->blue.offset = 0;
			break;
		case OSDM_RGB16_565_BGR:
			var->red.length = 5;	
			var->red.offset = 0;	
			var->green.length = 6;	
			var->green.offset = 5;	
			var->blue.length = 5;
			var->blue.offset = 11;
			break;
		case OSDM_DISPLAY_RGB24_888_RGB:
			var->red.length = 8;	
			var->red.offset = 16;	
			var->green.length = 8;	
			var->green.offset = 8;	
			var->blue.length = 8;
			var->blue.offset = 0;
			break;
		case OSDM_DISPLAY_RGB24_888_RBG:
			var->red.length = 8;	
			var->red.offset = 16;	
			var->green.length = 8;	
			var->green.offset = 0;	
			var->blue.length = 8;
			var->blue.offset = 8;
			break;
		case OSDM_DISPLAY_RGB24_888_GRB:
			var->red.length = 8;	
			var->red.offset = 16;	
			var->green.length = 8;	
			var->green.offset = 8;	
			var->blue.length = 8;
			var->blue.offset = 0;
			break;
		case OSDM_DISPLAY_RGB24_888_GBR:
			var->red.length = 8;	
			var->red.offset = 0;	
			var->green.length = 8;	
			var->green.offset = 8;	
			var->blue.length = 8;
			var->blue.offset = 16;
			break;
		case OSDM_DISPLAY_RGB24_888_BRG:
			var->red.length = 8;	
			var->red.offset = 8;	
			var->green.length = 8;	
			var->green.offset = 0;	
			var->blue.length = 8;
			var->blue.offset = 16;
			break;
		case OSDM_DISPLAY_RGB24_888_BGR:
			var->red.length = 8;	
			var->red.offset = 0;	
			var->green.length = 8;	
			var->green.offset = 8;	
			var->blue.length = 8;
			var->blue.offset = 16;
			break;
		case OSDM_DISPLAY_RGB16_565_RGB:
			var->red.length = 5;	
			var->red.offset = 11;	
			var->green.length = 6;	
			var->green.offset = 5;	
			var->blue.length = 5;
			var->blue.offset = 0;
			break;
		case OSDM_DISPLAY_RGB16_565_BGR:
			var->red.length = 5;	
			var->red.offset = 11;	
			var->green.length = 6;	
			var->green.offset = 5;	
			var->blue.length = 5;
			var->blue.offset = 0;
			break;
		default:
			return -1;
	}

	return 0;
}

static int dmw96osdm_copy_plane(struct dmw96plane *plane_dest, struct dmw96plane *plane_src)
{
	if (!plane_dest || !plane_src)
		return -1;

	plane_dest->info.var.bits_per_pixel = plane_src->info.var.bits_per_pixel;
	plane_dest->info.var.xres = plane_src->info.var.xres;
	plane_dest->info.var.yres = plane_src->info.var.yres;
	plane_dest->info.var.rotate = plane_src->info.var.rotate;
	plane_dest->info.par = plane_src->info.par;

	plane_dest->info.fix.line_length = plane_src->info.fix.line_length;
	plane_dest->info.fix.smem_start = plane_src->info.fix.smem_start;

	plane_dest->config = plane_src->config;
	memcpy(&plane_dest->pos, &plane_src->pos, sizeof(struct osdm_position));
	plane_dest->enb = plane_src->enb;
	plane_dest->commited = plane_src->commited;
	plane_dest->chroma_enb = plane_src->chroma_enb;
	plane_dest->per_pixel_alpha_enb = plane_src->per_pixel_alpha_enb;
	memcpy(&plane_dest->mask, &plane_src->mask , sizeof(struct osdm_mask ));
	plane_dest->per_plane_alpha = plane_src->per_plane_alpha;
	memcpy(&plane_dest->chroma_key, &plane_src->chroma_key , sizeof(osdm_rgb888));
	memcpy(&plane_dest->byte_order, &plane_src->byte_order , sizeof(plane_dest->byte_order)  );
	plane_dest->cur_phy_addr = plane_src->cur_phy_addr;
	plane_dest->offset = plane_src->offset;
	
	/*Transparency setting*/
	memcpy(&plane_dest->t_dim, &plane_src->t_dim, sizeof(struct fb_var_screeninfo));
	memcpy(&plane_dest->t_pos, &plane_src->t_pos, sizeof(struct osdm_position ));
	memcpy(&plane_dest->t_pixel_color, &plane_src->t_pixel_color, sizeof(osdm_abgr8888));

	/*Relevant to Video planes only*/
	memcpy(&plane_dest->video_color_control, &plane_src->video_color_control , sizeof(struct osdm_video_color_control));
	memcpy(&plane_dest->video_limits, &plane_src->video_limits , sizeof(struct osdm_video_limits ));

	/*Relevant to OSD planes only*/
	//memcpy(plane_dest->pal_red, plane_src->pal_red, sizeof(plane_dest->pal_red));
	//memcpy(plane_dest->pal_blue, plane_src->pal_blue, sizeof(plane_dest->pal_blue) );
	//memcpy(plane_dest->pal_green, plane_src->pal_green, sizeof(plane_dest->pal_green));
	//memcpy(plane_dest->pal_transp, plane_src->pal_transp, sizeof(plane_dest->pal_transp) );

	plane_dest->pal_size = plane_src->pal_size;
	plane_dest->lut_base = plane_src->lut_base;

	plane_dest->bg_color = plane_src->bg_color;
	plane_dest->bg_enb = plane_src->bg_enb;  
	plane_dest->dithering = plane_src->dithering;
	plane_dest->pause =plane_src->pause;   
	plane_dest->loop_mode = plane_src->loop_mode;

	plane_dest->vidmem = plane_src->vidmem;
#ifdef CONFIG_MACH_VERSATILE_BROADTILE
	plane_dest->smem_start = plane_src->smem_start;
#endif
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	plane_dest->dma_addr =  plane_src->dma_addr;
#endif
	plane_dest->rotate_mem = plane_src->rotate_mem;

	memcpy(plane_dest->list, plane_src->list , sizeof(*plane_dest->list) );
	
	memcpy(plane_dest->src , plane_src->src, sizeof(plane_dest->src));
	memcpy(plane_dest->dst , plane_src->dst, sizeof(plane_dest->dst));
	
	memcpy(&plane_dest->rotate_mem , &plane_src->rotate_mem, sizeof(plane_dest->rotate_mem));
	
	/*This semaphore is for mutual exlusion when 2 different process try to set same plane*/
	memcpy(&plane_dest->setting_sem , &plane_src->setting_sem, sizeof(plane_dest->setting_sem));

	return 0;
}

static void dmw96osdm_reset_plane(struct dmw96plane *osdm_plane)
{
	osdm_plane->config = 0;
	memset(&osdm_plane->pos, 0, sizeof(struct osdm_position));
	osdm_plane->enb = 0;
	osdm_plane->commited = 0;
	osdm_plane->chroma_enb = 0;
	osdm_plane->per_pixel_alpha_enb = 0;
	memset(&osdm_plane->mask, 0, sizeof(struct osdm_mask ));
	osdm_plane->per_plane_alpha = 255;
	memset(&osdm_plane->chroma_key, 0, sizeof(osdm_rgb888));
	memset(&osdm_plane->byte_order, 0, sizeof(osdm_plane->byte_order)  );
	osdm_plane->lut_base = 0;

	/*Transparency setting*/
	memset(&osdm_plane->t_dim, 0, sizeof(struct fb_var_screeninfo));
	memset(&osdm_plane->t_pos, 0, sizeof(struct osdm_position ));
	memset(&osdm_plane->t_pixel_color, 0, sizeof(osdm_abgr8888));

	/*Relevant to Video planes only*/
	memset(&osdm_plane->video_color_control, 0, sizeof(struct osdm_video_color_control));
	osdm_plane->video_color_control.brightness = 0;
	osdm_plane->video_color_control.contrast = 64;
	osdm_plane->video_color_control.saturation = 64;
	osdm_plane->video_color_control.hue = 0;

	memset(&osdm_plane->video_limits, 0, sizeof(struct osdm_video_limits ));
	osdm_plane->video_limits.chroma_max = 240;
	osdm_plane->video_limits.chroma_min = 16;
	osdm_plane->video_limits.luma_max = 235;
	osdm_plane->video_limits.luma_min = 16;

	/*Relevant to OSD planes only*/
	memset(osdm_plane->pal_red, 0, sizeof(osdm_plane->pal_red));
	memset(osdm_plane->pal_blue, 0, sizeof(osdm_plane->pal_blue) );
	memset(osdm_plane->pal_green, 0, sizeof(osdm_plane->pal_green));
	memset(osdm_plane->pal_transp, 0, sizeof(osdm_plane->pal_transp) );

	osdm_plane->pal_size = 0;
}

static void dmw96osdm_reset_settings(void)
{
	int i = 0;
	PDEBUG_START

	if (!g_osdm)
		return;

	for (i=0 ; i < NUM_OF_IN_PLANES ; i++)
		dmw96osdm_reset_plane(&g_osdm->planes[i]);
	
	for (i=0 ; i < NUM_OF_IN_PLANES ; i++)
		dmw96osdm_reset_plane(&g_osdm->uncommitted_planes[i]);	

	PDEBUG_END
}

static int dmw96osdm_check_sizes(struct fb_info *info , unsigned int osdm_type)
{
	struct dmw96osdm* osdm = FB2OOSDM(info);
    int bpp = 0;
	int total_byte_size = 0;
	unsigned int rotate = 1;

	if (!osdm  || osdm_type > DISPLAY_PLANE)
		return -1;
	
	if (osdm->uncommitted_planes[osdm_type].info.var.rotate != 0)
		rotate = 2;
	
	bpp = dmw96osdm_osdm_get_bpp(osdm->uncommitted_planes[osdm_type].config);
	if (bpp == 0 || osdm->uncommitted_planes[osdm_type].info.var.xres == 0 || osdm->uncommitted_planes[osdm_type].info.var.yres == 0) {
		PDEBUG("Invalid image configure since it's bpp is %d or invalid xsize %d or invalid ysize %d\n", bpp ,
			   osdm->uncommitted_planes[osdm_type].info.var.xres , osdm->uncommitted_planes[osdm_type].info.var.yres);
		return FAIL;
	}

	total_byte_size = osdm->uncommitted_planes[osdm_type].info.var.xres * osdm->uncommitted_planes[osdm_type].info.var.yres * bpp * rotate / 8;
	if (dmw96_osdm_planes_sizes[osdm_type] > 0 && total_byte_size >= dmw96_osdm_planes_sizes[osdm_type] * 1024 ) {
		PDEBUG("Allocated memory for frame buffer %s is 0 OR image size %d is bigger or equal to pre allocated memory size %d\n", plane_name_arr[osdm_type] , total_byte_size , dmw96_osdm_planes_sizes[osdm_type] * 1024);
		return FAIL;
	}

	return SUCCESS;
}

#ifdef CONFIG_FB_DMW96_OSDM_ROTATE

static void print_image(struct dmw96dma_image* img)
{
	if (img) {
		PDEBUG("img->format = %d\n",img->format);
		PDEBUG("img->coherent = %x\n",img->coherent);
		PDEBUG("img->width = %d\n", img->width);
		PDEBUG("img->height = %d\n",img->height);
		PDEBUG("img->data.coherent = 0x%x\n",img->data.coherent);
	}
}

static unsigned int setup_image(struct dmw96dma_image* img , int width, int height, enum dmw96dma_width fmt, unsigned int phys_addr)
{
	if (!img)
		return -1;
	
	img->format = fmt;
	img->coherent = 1;
	img->width = width;
	img->height = height;
	img->data.coherent = phys_addr;
		
	return 0;
}

static int calc_y_rgb_image_addr(unsigned int xres, unsigned int yres, 
								unsigned int src_rgb_y_add, unsigned int dst_rgb_y_add,
								unsigned int offset,
								struct dmw96dma_image* src , struct dmw96dma_image* dst,
								enum dmw96dma_width fmt,
								unsigned int degree)
{
	unsigned int _xres = 0, _yres = 0;

	if (!src || !dst)
		return -1;

	check_rotate_and_swap(xres , yres , &_xres , &_yres , degree);
	
	/*Setting up src setup of the first buffer*/
	setup_image(&src[0] , xres ,yres , 		fmt, src_rgb_y_add);
	/*Setting up dst setup of the first buffer*/
	setup_image(&dst[0] , _xres , _yres , 		fmt, dst_rgb_y_add);
	/*Setting up src setup of the second buffer*/
	setup_image(&src[3] , xres , yres ,    fmt, src_rgb_y_add + offset);
	/*Setting up dst setup of the second buffer*/
	setup_image(&dst[3] ,  _xres , _yres ,    fmt, dst_rgb_y_add + offset);

	return 0;
}

static int calc_uv_image_addr(unsigned int xres, unsigned int yres, 
								  unsigned int src_u_add, unsigned int src_v_add,
								  unsigned int dst_u_add, unsigned int dst_v_add,
								  unsigned int offset,
								  struct dmw96dma_image* src , struct dmw96dma_image* dst,
								  enum dmw96dma_width fmt,
								  unsigned int is_planar,
								  unsigned int degree)
{
	unsigned int _xres = 0;
	unsigned int _yres = 0;

	if (!src || !dst)
		return -1;

	check_rotate_and_swap(xres , yres , &_xres , &_yres , degree);

	/*Setting up src setup of the first buffer*/
	setup_image(&src[1] ,xres ,yres ,fmt ,src_u_add);

	/*Setting up dst setup of the first buffer*/
	setup_image(&dst[1] ,_xres ,_yres ,fmt ,dst_u_add);    

	/*Setting up src setup of the second buffer*/
	setup_image(&src[4] ,xres ,yres ,fmt ,src_u_add + offset);     

	/*Setting up dst setup of the second buffer*/
	setup_image(&dst[4] ,_xres ,_yres ,fmt ,dst_u_add + offset);    

	if (is_planar) {
		/*Setting up src setup of the first buffer*/
		setup_image(&src[2] ,xres ,yres ,fmt ,src_v_add);
	
		/*Setting up dst setup of the first buffer*/
		setup_image(&dst[2] ,_xres ,_yres ,fmt ,dst_v_add);    
	
		/*Setting up src setup of the second buffer*/
		setup_image(&src[5] ,xres ,yres ,fmt ,src_v_add + offset);     
	
		/*Setting up dst setup of the second buffer*/
		setup_image(&dst[5] ,_xres ,_yres ,fmt ,dst_v_add + offset);
	}

	return 0;
}


static int get_src_dst_image_config(struct fb_info *info , unsigned int osdm_type , struct dmw96dma_image* src , struct dmw96dma_image* dst)
{
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int src_rgb_y_add = 0, src_u_add = 0, src_v_add = 0;
	unsigned int dst_rgb_y_add = 0, dst_u_add = 0, dst_v_add = 0;
	unsigned int offset = 0;
	unsigned int xres = 0, yres=0 ;
	unsigned int degree = 0;
	unsigned int input_addr = 0;

	if (!osdm || osdm_type > DISPLAY_PLANE)
		return -1;

	if (osdm->uncommitted_planes[osdm_type].info.var.rotate == 90)
		degree = DMW96DMA_ROTATE_90_CW;
	else if (osdm->uncommitted_planes[osdm_type].info.var.rotate == 180)
		degree = DMW96DMA_ROTATE_180_CW;
	else if(osdm->uncommitted_planes[osdm_type].info.var.rotate == 270)
		degree = DMW96DMA_ROTATE_90_CCW;
	else
		return -1;

	if (osdm->uncommitted_planes[osdm_type].info.var.reserved[1] != 0)
		input_addr = osdm->uncommitted_planes[osdm_type].info.var.reserved[1];
	else
		input_addr = osdm->uncommitted_planes[osdm_type].info.fix.smem_start;

	src_rgb_y_add = DMW96OSDM_GET_Y_RGB_ADDRESS(osdm->uncommitted_planes[osdm_type].info.var.xres, osdm->uncommitted_planes[osdm_type].info.var.yres , osdm->uncommitted_planes[osdm_type].config , input_addr);
	src_u_add = DMW96OSDM_GET_U_UV_ADDRESS(osdm->uncommitted_planes[osdm_type].info.var.xres, osdm->uncommitted_planes[osdm_type].info.var.yres , osdm->uncommitted_planes[osdm_type].config , input_addr);
	src_v_add = DMW96OSDM_GET_V_ADDRESS(osdm->uncommitted_planes[osdm_type].info.var.xres, osdm->uncommitted_planes[osdm_type].info.var.yres , osdm->uncommitted_planes[osdm_type].config , input_addr);

	dst_rgb_y_add = DMW96OSDM_GET_Y_RGB_ADDRESS(osdm->uncommitted_planes[osdm_type].info.var.xres, osdm->uncommitted_planes[osdm_type].info.var.yres , osdm->uncommitted_planes[osdm_type].config , osdm->uncommitted_planes[osdm_type].rotate_mem );
	dst_u_add =	DMW96OSDM_GET_U_UV_ADDRESS(osdm->uncommitted_planes[osdm_type].info.var.xres, osdm->uncommitted_planes[osdm_type].info.var.yres , osdm->uncommitted_planes[osdm_type].config , osdm->uncommitted_planes[osdm_type].rotate_mem );
	dst_v_add = DMW96OSDM_GET_V_ADDRESS(osdm->uncommitted_planes[osdm_type].info.var.xres, osdm->uncommitted_planes[osdm_type].info.var.yres , osdm->uncommitted_planes[osdm_type].config , osdm->uncommitted_planes[osdm_type].rotate_mem );

	offset = (osdm->uncommitted_planes[osdm_type].info.var.yres * osdm->uncommitted_planes[osdm_type].info.var.xres * info->var.bits_per_pixel) / 8;

	xres = osdm->uncommitted_planes[osdm_type].info.var.xres;
	yres = osdm->uncommitted_planes[osdm_type].info.var.yres;

	if (IS_PLANAR(osdm->uncommitted_planes[osdm_type].config) || IS_SEMIPLANAR(osdm->uncommitted_planes[osdm_type].config)) {
		calc_y_rgb_image_addr(xres, yres, 
							  src_rgb_y_add,
							  dst_rgb_y_add,
							  offset,
							  src , dst,
							  DMW96DMA_WIDTH_8,
							  osdm->uncommitted_planes[osdm_type].info.var.rotate);
		if (IS_PLANAR(osdm->uncommitted_planes[osdm_type].config)) {
			if (IS_VIDEO_420(osdm->uncommitted_planes[osdm_type].config))
				calc_uv_image_addr(xres/2, yres/2, 
									  src_u_add, src_v_add,
									  dst_u_add, dst_v_add,
									  offset,
									  src , dst,
									  DMW96DMA_WIDTH_8,
									  1,
									  osdm->uncommitted_planes[osdm_type].info.var.rotate);
			else if (IS_VIDEO_444(osdm->uncommitted_planes[osdm_type].config) )
				calc_uv_image_addr(xres, yres, 
									  src_u_add, src_v_add,
									  dst_u_add, dst_v_add,
									  offset,
									  src , dst,
									  DMW96DMA_WIDTH_8,
									  1,
									  osdm->uncommitted_planes[osdm_type].info.var.rotate);
			else
				goto invalid_fmt_rotate;
		} 
		else if (IS_SEMIPLANAR(osdm->uncommitted_planes[osdm_type].config)) {
			if (IS_VIDEO_420(osdm->uncommitted_planes[osdm_type].config))
				calc_uv_image_addr(xres/2, yres/2, 
									  src_u_add, src_v_add,
									  dst_u_add, dst_v_add,
									  offset,
									  src , dst,
									  DMW96DMA_WIDTH_16,
									  0,
									  osdm->uncommitted_planes[osdm_type].info.var.rotate);
			else if (IS_VIDEO_444(osdm->uncommitted_planes[osdm_type].config) )
				calc_uv_image_addr(xres, yres, 
									  src_u_add, src_v_add,
									  dst_u_add, dst_v_add,
									  offset,
									  src , dst,
									  DMW96DMA_WIDTH_16,
									  0,
									  osdm->uncommitted_planes[osdm_type].info.var.rotate);
			else
				goto invalid_fmt_rotate;
		}
	}
	else if (IS_PACKED(osdm->uncommitted_planes[osdm_type].config)) {
		if (IS_RGB32(osdm->uncommitted_planes[osdm_type].config))
			calc_y_rgb_image_addr(xres, yres, 
								  src_rgb_y_add,
								  dst_rgb_y_add,
								  offset,
								  src , dst,
								  DMW96DMA_WIDTH_32,
								  osdm->uncommitted_planes[osdm_type].info.var.rotate);
		else if (IS_RGB16(osdm->uncommitted_planes[osdm_type].config))
			calc_y_rgb_image_addr(xres, yres, 
								  src_rgb_y_add,
								  dst_rgb_y_add,
								  offset,
								  src , dst,
								  DMW96DMA_WIDTH_16,
								  osdm->uncommitted_planes[osdm_type].info.var.rotate);
		else if (IS_8BPP(osdm->uncommitted_planes[osdm_type].config))
			calc_y_rgb_image_addr(xres, yres, 
								  src_rgb_y_add,
								  dst_rgb_y_add,
								  offset,
								  src , dst,
								  DMW96DMA_WIDTH_8,
								  osdm->uncommitted_planes[osdm_type].info.var.rotate);
		else
			goto invalid_fmt_rotate;
	}

	return 0;

invalid_fmt_rotate:
	PDEBUG("Invalid format %x for rotate\n",osdm->uncommitted_planes[osdm_type].config);
	return -1;
}

static int copy_uncommitted_and_enbale_lcdc_vsync_int(struct fb_info *info)
{
	int plane_num = -1;
	unsigned long flags;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	PDEBUG_START

	if (!osdm )
		return -1;

	plane_num = get_plane_num(osdm,info);

	spin_lock_irqsave(osdm->copy_lock, flags);

	dmw96osdm_copy_plane(&osdm->planes[plane_num], &osdm->uncommitted_planes[plane_num]);
	
	osdm->planes[plane_num].commited = 1;

	if (!osdm->is_working)
		lcdc_enable_vsync_int_once();

	spin_unlock_irqrestore(osdm->copy_lock, flags);
	PDEBUG_END

	return 0;
}

static void rotate_image_finish(int status, void *context)
{
	int plane_num = -1;
	struct fb_info *info = (struct fb_info *) context;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	plane_num = get_plane_num(osdm,info);
	if (status)
		PDEBUG("DMA rotate failed: %d\n", status);
	else
		PDEBUG("DMA rotate success\n");

	osdm->uncommitted_planes[plane_num].cur_phy_addr = osdm->uncommitted_planes[plane_num].rotate_mem + osdm->uncommitted_planes[plane_num].offset;

	copy_uncommitted_and_enbale_lcdc_vsync_int(info);

	up(&osdm->uncommitted_planes[plane_num].rotate_sem);	
}

static int free_rotate_dma_list(unsigned int osdm_type, struct dmw96osdm *osdm)
{
	if (!osdm)
		return -1;

	if (osdm->uncommitted_planes[osdm_type].list[0]) {
		dmw96dma_free_list(osdm->uncommitted_planes[osdm_type].list[0]);
		osdm->uncommitted_planes[osdm_type].list[0] = NULL;
	}

	if (osdm->uncommitted_planes[osdm_type].list[1]) {
		dmw96dma_free_list(osdm->uncommitted_planes[osdm_type].list[1]);
		osdm->uncommitted_planes[osdm_type].list[1] = NULL;
	}

	return 0;
}

static int alloc_rotate_dma_list(unsigned int osdm_type, struct fb_info *info )
{
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm)
		return -1;

	free_rotate_dma_list(osdm_type , osdm);

	/*One list is for the first buffer*/
	osdm->uncommitted_planes[osdm_type].list[0] = dmw96dma_alloc_image_list(rotate_image_finish, info );
	if (!osdm->uncommitted_planes[osdm_type].list[0])
		return -ENOMEM;

	/*Second list is for the first buffer*/
	osdm->uncommitted_planes[osdm_type].list[1] = dmw96dma_alloc_image_list(rotate_image_finish, info);
	if (!osdm->uncommitted_planes[osdm_type].list[1])
		return -ENOMEM;
	
	return 0;
}

void print_src_dst_image( struct dmw96dma_image* src , struct dmw96dma_image* dst)
{
	unsigned int i = 0;

	PDEBUG("*****************SRC images********************\n");

	for (i=0 ; i<6 ; i++) {
		PDEBUG("*****image [%d]*****\n", i);
		print_image(&src[i]);
	}

	PDEBUG("***********************************************\n");
	PDEBUG("*****************DST images********************\n");

	for (i=0 ; i<6 ; i++) {
		PDEBUG("*****image [%d]*****\n", i);
		print_image(&dst[i]);
	}

	PDEBUG("***********************************************\n");
}

static int dmw96dma_add_rotate_operation(struct fb_info *info , unsigned int osdm_type)
{
	unsigned int res_a = SUCCESS;
	unsigned int res_b = SUCCESS;
	unsigned int degree = 0;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm || osdm_type > DISPLAY_PLANE)
		return -1;

	if (osdm->uncommitted_planes[osdm_type].info.var.rotate == 90)
		degree = DMW96DMA_ROTATE_90_CW;
	else if (osdm->uncommitted_planes[osdm_type].info.var.rotate == 180)
		degree = DMW96DMA_ROTATE_180_CW;
	else if(osdm->uncommitted_planes[osdm_type].info.var.rotate == 270)
		degree = DMW96DMA_ROTATE_90_CCW;
	else
		return -1;

	memset(osdm->uncommitted_planes[osdm_type].src, 0, sizeof(osdm->uncommitted_planes[osdm_type].src));
	memset(osdm->uncommitted_planes[osdm_type].dst, 0, sizeof(osdm->uncommitted_planes[osdm_type].dst));

	res_a = alloc_rotate_dma_list(osdm_type, info);
	if (FAILED(res_a))
		goto free_dma_list_rotate;

	res_a = get_src_dst_image_config(info, osdm_type, osdm->uncommitted_planes[osdm_type].src, osdm->uncommitted_planes[osdm_type].dst);
	if (FAILED(res_a))
		goto free_dma_list_rotate;

	print_src_dst_image(osdm->uncommitted_planes[osdm_type].src, osdm->uncommitted_planes[osdm_type].dst);

	PDEBUG("Add rotate for Y or RGB\n");
	res_a = dmw96dma_add_rotate(osdm->uncommitted_planes[osdm_type].list[0],
								&osdm->uncommitted_planes[osdm_type].src[0], 
								&osdm->uncommitted_planes[osdm_type].dst[0],
								NULL, NULL, degree, NULL, NULL);

	res_b = dmw96dma_add_rotate(osdm->uncommitted_planes[osdm_type].list[1],
								&osdm->uncommitted_planes[osdm_type].src[3],
								&osdm->uncommitted_planes[osdm_type].dst[3],
								NULL, NULL, degree, NULL, NULL);
	
	if (FAILED(res_a) || FAILED(res_b)) {
		PDEBUG("Error adding to list hence perform free list res_a=0x%x res_b=0x%x\n",res_a ,res_b);
		goto free_dma_list_rotate;
	}

	if (osdm->uncommitted_planes[osdm_type].src[1].height != 0 && osdm->uncommitted_planes[osdm_type].src[1].width != 0) {
		PDEBUG("Add rotate for UV or U\n");
		res_a = dmw96dma_add_rotate(osdm->uncommitted_planes[osdm_type].list[0],
									&osdm->uncommitted_planes[osdm_type].src[1], 
									&osdm->uncommitted_planes[osdm_type].dst[1],
									NULL, NULL, degree, NULL, NULL);

		res_b = dmw96dma_add_rotate(osdm->uncommitted_planes[osdm_type].list[1],
									&osdm->uncommitted_planes[osdm_type].src[4],
									&osdm->uncommitted_planes[osdm_type].dst[4],
									NULL, NULL, degree, NULL, NULL);
	}
	
	if (FAILED(res_a) || FAILED(res_b)) 
		goto free_dma_list_rotate;

	if (osdm->uncommitted_planes[osdm_type].src[2].height != 0 && osdm->uncommitted_planes[osdm_type].src[2].width != 0) {
		PDEBUG("Add rotate for V\n");
		res_a = dmw96dma_add_rotate(osdm->uncommitted_planes[osdm_type].list[0],
									&osdm->uncommitted_planes[osdm_type].src[2],
									&osdm->uncommitted_planes[osdm_type].dst[2],
									NULL, NULL, degree, NULL, NULL);

		res_b = dmw96dma_add_rotate(osdm->uncommitted_planes[osdm_type].list[1],
									&osdm->uncommitted_planes[osdm_type].src[5],
									&osdm->uncommitted_planes[osdm_type].dst[5],
									NULL, NULL, degree, NULL, NULL);
	}

	if (FAILED(res_a) || FAILED(res_b)) 
		goto free_dma_list_rotate;

	PDEBUG("Success add to rotate to dma list\n");
	return 0;

free_dma_list_rotate:
	printk("Fail to add to rotate to dma HW (Check image sizes, format, address fulfil DMA limitation) relesing the DMA list\n");
	free_rotate_dma_list(osdm_type,osdm);
	return ( res_a | res_b );
}
#endif

static int dmw96osdm_plane_rotate_set(struct fb_info *info , unsigned int osdm_type) 
{
	unsigned int res = SUCCESS;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm || osdm_type > DISPLAY_PLANE)
		return -1;

	if (osdm->uncommitted_planes[osdm_type].info.var.rotate != 90 && osdm->uncommitted_planes[osdm_type].info.var.rotate != 180 && 
		 osdm->uncommitted_planes[osdm_type].info.var.rotate != 270 && osdm->uncommitted_planes[osdm_type].info.var.rotate != 0) {	
		printk("OSDM invalid rotate angle %d \n", osdm->uncommitted_planes[osdm_type].info.var.rotate);
		return -1;
	}

	if ( osdm->uncommitted_planes[osdm_type].info.var.rotate == 90 || osdm->uncommitted_planes[osdm_type].info.var.rotate == 180 || 
		 osdm->uncommitted_planes[osdm_type].info.var.rotate == 270 ) {
#ifndef CONFIG_FB_DMW96_OSDM_ROTATE
		printk("OSDM not support for rotate please enable rotate operation under OSDM at the kernel\n");
		return -1;
#endif
		if ( osdm->uncommitted_planes[osdm_type].config == OSDM_1_BPP_PALETTE ||
			 osdm->uncommitted_planes[osdm_type].config == OSDM_2_BPP_PALETTE ||
			 osdm->uncommitted_planes[osdm_type].config == OSDM_4_BPP_PALETTE ||
			 IS_VIDEO_422(osdm->uncommitted_planes[osdm_type].config)) {
			printk("OSDM GDMA is not able to preform rotare to 4bpp, 2bpp or 1bpp or any 4:2:2 formats\n");
			return -1;
		}
	}
	if (SUCCEEDED(res)) {
#ifdef CONFIG_FB_DMW96_OSDM_ROTATE
		if (osdm->uncommitted_planes[osdm_type].info.var.rotate != 0)
			res = dmw96dma_add_rotate_operation(&osdm->uncommitted_planes[osdm_type].info,osdm_type);
#endif
	}
		
	return res;
}

static int  dmw96osdm_common_settings(struct fb_info *info , unsigned int osdm_type) 
{
    unsigned int res = SUCCESS;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm || osdm_type > DISPLAY_PLANE)
		return -1;

	PDEBUG_START;

	res = dmw96osdm_plane_transparent_block_pos_set(osdm->planes[osdm_type].config,
													&osdm->planes[osdm_type].t_pos,
													&osdm->planes[osdm_type].info,
													osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_transparent_block_dim_set(osdm->planes[osdm_type].config,
													&osdm->planes[osdm_type].t_dim,
													&osdm->planes[osdm_type].info,
													osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_transparent_pixel_set(osdm->planes[osdm_type].t_pixel_color,
												osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_chroma_enb(osdm->planes[osdm_type].chroma_enb,
									 osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_chromakey_set(osdm->planes[osdm_type].chroma_key,
										osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_per_pixel_alpha_enb(osdm->planes[osdm_type].per_pixel_alpha_enb,
											  osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_alpha_set(osdm->planes[osdm_type].per_plane_alpha,
									osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_masking_set(osdm->planes[osdm_type].mask,
									  osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_address_set(osdm->planes[osdm_type].info.var.xres,
									  osdm->planes[osdm_type].info.var.yres,
									  osdm->planes[osdm_type].cur_phy_addr,
									  osdm->planes[osdm_type].config,
									  osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_set(&osdm->planes[DISPLAY_PLANE].info,
							  &osdm->planes[osdm_type].info,
							  &osdm->planes[osdm_type].pos,
							  osdm->planes[osdm_type].config,
							  osdm_type);
	if (FAILED(res)) 
		return res;
	
	PDEBUG_END;
					
    return res;
}

static int dmw96osdm_video_settings(struct fb_info *info , unsigned int osdm_type)
{
    int res = SUCCESS;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm || osdm_type > DISPLAY_PLANE)
		return -1;
	
	res = dmw96osdm_color_video_control_set(&osdm->planes[osdm_type].video_color_control, osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_dma_channel_byteorder_set(osdm->planes[osdm_type].byte_order[0], 0, osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_dma_channel_byteorder_set(osdm->planes[osdm_type].byte_order[1], 1, osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_dma_channel_byteorder_set(osdm->planes[osdm_type].byte_order[2], 2, osdm_type);
	if (FAILED(res)) 
		return res;
	
    return res;
}

static int  dmw96osdm_osd_settings(struct fb_info *info , unsigned int osdm_type)
{
    struct fb_cmap cmap;
	int res = SUCCESS;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm || osdm_type > DISPLAY_PLANE)
		return -1;

	memset(&cmap,0,sizeof(struct fb_cmap));

	cmap.red = osdm->planes[osdm_type].pal_red;
	cmap.green = osdm->planes[osdm_type].pal_green;
	cmap.blue = osdm->planes[osdm_type].pal_blue;
	cmap.transp = osdm->planes[osdm_type].pal_transp;
	cmap.len = osdm->planes[osdm_type].pal_size;
    
    res = dmw96osdm_osd_palette_set(&cmap, osdm->planes[osdm_type].lut_base ,osdm_type);
	if (SUCCEEDED(res)) {
		res = dmw96osdm_plane_dma_channel_byteorder_set(osdm->planes[osdm_type].byte_order[0], 0 , osdm_type);
	}

	return res;
}

static int dmw96osdm_display_settings(struct dmw96osdm* osdm)
{
	int res = SUCCESS;

	if (!osdm)
		return -1;
	
	if (osdm->planes[DISPLAY_PLANE].commited) {
		/* We save the config of the pannel under the reserved field */
        res = dmw96osdm_display_set(&osdm->planes[DISPLAY_PLANE].info,
									osdm->planes[DISPLAY_PLANE].info.var.reserved[0]);

		if (SUCCEEDED(res)) {
			/* Set the display properties */
			dmw96osdm_display_dithering_set(osdm->planes[DISPLAY_PLANE].dithering);
			dmw96osdm_display_background_set(osdm->planes[DISPLAY_PLANE].bg_color);
			dmw96osdm_display_background_enable(osdm->planes[DISPLAY_PLANE].bg_enb);
		}
		
		osdm->planes[DISPLAY_PLANE].commited = 0;
	}

	return res;
}

static int dmw96osdm_plane_settings(struct fb_info *info, unsigned int plane) 
{ 
	int res = SUCCESS;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm)
		return -1;

	PDEBUG_START

	res = dmw96osdm_common_settings(info,plane);
	if (FAILED(res)) 
		return res;

	if (plane == MAIN_VIDEO_PLANE || plane == SUB_VIDEO_PLANE) {
		res = dmw96osdm_video_settings(info,plane);
		if (FAILED(res)) 
			return res;
	}
	else {
		res = dmw96osdm_osd_settings(info,plane);
		if (FAILED(res)) 
			return res;
	}

	PDEBUG_END

	return res;
}

static int dmw96osdm_check_plane_settings(struct fb_info *info, unsigned int osdm_type) 
{
	int res = SUCCESS;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	PDEBUG_START

	if (!osdm)
		return -1;

	if (osdm_type != MAIN_VIDEO_PLANE && osdm_type != SUB_VIDEO_PLANE && 
		osdm_type != MAIN_OSD_PLANE && osdm_type != SUB_OSD_PLANE_1 && 
		osdm_type != SUB_OSD_PLANE_2 && osdm_type != DISPLAY_PLANE)
		return -1;

	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(osdm->uncommitted_planes[osdm_type].config)) 
		   && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(osdm->uncommitted_planes[osdm_type].config)) ) ) {
		PDEBUG("Invalid osdm type or configuration\n");
		return FAIL;
	}

	res = dmw96osdm_check_sizes(info,osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_transparent_block_pos_test(osdm->uncommitted_planes[osdm_type].config,
													 &osdm->uncommitted_planes[osdm_type].t_pos,
													 &osdm->uncommitted_planes[osdm_type].info,
													 osdm_type);
	if (FAILED(res))
		return res;

	res = dmw96osdm_plane_transparent_block_dim_test(osdm->uncommitted_planes[osdm_type].config,
													 &osdm->uncommitted_planes[osdm_type].t_dim,
													 &osdm->uncommitted_planes[osdm_type].info,
													 osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_masking_test(osdm->uncommitted_planes[osdm_type].mask, osdm_type);
	if (FAILED(res)) 
		return res;

	res = dmw96osdm_plane_test(&osdm->planes[DISPLAY_PLANE].info,
							   &osdm->uncommitted_planes[osdm_type].info,
							   &osdm->uncommitted_planes[osdm_type].pos,
							   osdm->uncommitted_planes[osdm_type].config,
							   osdm_type);
	if (FAILED(res)) 
		return res;

	PDEBUG_END

	return 0;
}

static int dmw96osdm_planes_settings(void) 
{
	int res = SUCCESS;
	unsigned int plane_num = 0;
	
	if (!g_osdm)
		return -1;

	PDEBUG_START

	res = dmw96osdm_display_settings(g_osdm);
	if (FAILED(res)) 
		return res;

	for (plane_num = 0 ; plane_num < NUM_OF_IN_PLANES ; plane_num++) 
		if (g_osdm->planes[plane_num].enb && g_osdm->planes[plane_num].commited) {
				dmw96_print_plane(&g_osdm->planes[plane_num].info, plane_num);
				res = dmw96osdm_plane_settings(&g_osdm->planes[plane_num].info, plane_num);
				if (FAILED(res)) {
					PDEBUG("%s %d Fail to perform single setting \n", __FUNCTION__, __LINE__);
					return res;
				}
				else {
					if (IS_PLANE_OSD(plane_num))
						g_osdm->planes[plane_num].just_committed = 1;
						
					PDEBUG("%s %d SUCCESS to perform single setting \n", __FUNCTION__, __LINE__);
				}
		}
	

	PDEBUG_END

	return res;
}

static void dmw96osdm_mix_pre_settings(void)
{
    unsigned int i = 0;
	
	if (!g_osdm)
		return;

	/*Disable the output layer*/
	dmw96osdm_display_enb_set(0);

    /*Disable all other layers*/
	for (i=0 ; i < NUM_OF_IN_PLANES ; i++) {
        dmw96osdm_plane_enb_set(0, i);
	}

	/*Disable the output DMA*/
	dmw96osdm_display_dma_enb_set(0);

	/*Disable all other DMA layers*/
	for (i=0 ; i < NUM_OF_IN_PLANES ; i++) {
		dmw96osdm_plane_dma_enb_set(0,g_osdm->planes[i].config ,i);
	}
}

static void dmw96osdm_mix_post_settings(void)
{
    unsigned int i = 0;
	unsigned int mix_enb = 0;

	if (!g_osdm)
		return;

	PDEBUG_START

	for (i=0 ; i < NUM_OF_IN_PLANES ; i++) {
		if (g_osdm->planes[i].enb)
			dmw96osdm_plane_enb_set(1, i);
	}

	/*Enable all other DMA layers*/
	for (i=0 ; i < NUM_OF_IN_PLANES ; i++) {
		if (g_osdm->planes[i].enb)
			dmw96osdm_plane_dma_enb_set(1,g_osdm->planes[i].config ,i);
	}

	for (i=0 ; i < NUM_OF_IN_PLANES ; i++)
		if (g_osdm->planes[i].commited) {
			g_osdm->is_working = 1;
			mix_enb = 1;
			dmw96osdm_display_dma_enb_set(1);
			dmw96osdm_display_enb_set(1);
			break;
		}

	if (!mix_enb) {
		clk_disable(g_osdm->clk);
		printk("Warning OSDM mix was not started!\n");
	}

	for (i=0 ; i < NUM_OF_IN_PLANES ; i++)
		g_osdm->planes[i].commited = 0;
	
	PDEBUG_END
}

int start_mix(void)
{
	int res = 0;
	unsigned long flags;

	PDEBUG_START

	clk_enable(g_osdm->clk);

	spin_lock_irqsave(g_osdm->copy_lock, flags);

	dmw96osdm_mix_pre_settings();

	res = dmw96osdm_planes_settings();
	if (SUCCEEDED(res)) 
	   dmw96osdm_mix_post_settings();
	else {
		printk("%s %d Error performing setting of plane\n", __FUNCTION__, __LINE__);
		dmw96osdm_reset_settings();
		clk_disable(g_osdm->clk);
		res = FAIL;
	}

	spin_unlock_irqrestore(g_osdm->copy_lock, flags);
	
#ifdef CONFIG_MACH_VERSATILE_BROADTILE
	while ( !( dmw96osdm_is_display_finished() ) ) { };
	dmw96osdm_mix_pre_settings();
	dmw96osdm_interrupt_handle();
#ifdef CONFIG_FB_DW74_DOUBLEBUF
	pan_osdm();
#endif

#endif

	PDEBUG_END

	return res;
}
EXPORT_SYMBOL(start_mix);

#ifdef CONFIG_MACH_VERSATILE_BROADTILE

static int allocate_osdm_buffers(struct dmw96osdm *osdm, struct device *dev)
{
	unsigned int i = 0;
	unsigned int address_offset = 0xe00c0000;
	unsigned int address_offset_t = 0;
	unsigned int rotate = 1;

#ifdef CONFIG_FB_DW74_DOUBLEBUF
		address_offset = 0xe0180000;
#endif
#ifdef CONFIG_FB_DMW96_OSDM_ROTATE
		rotate = 2;
#endif
	address_offset_t = address_offset;

	if (!osdm || !dev)
		return -1;
	
	for (i=0 ; i < NUM_OF_IN_PLANES ; i++) {
		if (dmw96_osdm_planes_sizes[i] != 0) {
			/*map plane's buffers*/
			if (!request_mem_region(address_offset_t, dmw96_osdm_planes_sizes[i] * 1024 * rotate, plane_name_arr[i])) {
				dev_err(dev, "cannot request mem region for %s \n", plane_name_arr[i]);
				return -EBUSY;
			}
			osdm->uncommitted_planes[i].vidmem = ioremap_nocache(address_offset_t, dmw96_osdm_planes_sizes[i] * 1024 * rotate);
			if (!osdm->uncommitted_planes[i].vidmem) {
				dev_err(dev, "cannot map sdram for %s\n", plane_name_arr[i]);
				goto osdm_release_vidmem;
			}
#ifdef CONFIG_FB_DMW96_OSDM_ROTATE
			osdm->uncommitted_planes[i].rotate_mem = address_offset_t + (dmw96_osdm_planes_sizes[i] * 1024);
			PDEBUG("%s rotate_mem = 0x%x\n", plane_name_arr[i], osdm->uncommitted_planes[i].rotate_mem);
#endif
			osdm->uncommitted_planes[i].smem_start = address_offset_t;
			PDEBUG("%s virtual mem = %p physical mem = 0x%lx\n",plane_name_arr[i],
																osdm->uncommitted_planes[i].vidmem,osdm->uncommitted_planes[i].smem_start);
			address_offset_t += dmw96_osdm_planes_sizes[i] * 1024 * rotate;
			/*align to 4KB*/
			address_offset_t += 4*1024 - 1;
			address_offset_t &= ~0xfff;
		}
	}
	return 0;

osdm_release_vidmem:
	address_offset_t = address_offset;
	for (i=0 ; i < NUM_OF_IN_PLANES ; i++) {
		if (dmw96_osdm_planes_sizes[i] != 0) {
			if (osdm->uncommitted_planes[i].vidmem != NULL)
			{
				release_mem_region(address_offset_t, dmw96_osdm_planes_sizes[i] * 1024 * rotate);
			}
		}
		address_offset_t += dmw96_osdm_planes_sizes[i] * 1024 * rotate;
	}
	return -EBUSY;
}

#else

static int allocate_osdm_buffers(struct dmw96osdm *osdm, struct device *dev)
{
	unsigned int i = 0;
    int ret=0;
	unsigned int rotate = 1;

	if (!osdm || !dev)
		return -1;
#ifdef CONFIG_FB_DMW96_OSDM_ROTATE
		rotate = 2;
#endif
	for (i=0 ; i < NUM_OF_IN_PLANES ; i++) {

		if (dmw96_osdm_planes_sizes[i] != 0) {
			osdm->uncommitted_planes[i].vidmem = dma_alloc_coherent(dev,
																	dmw96_osdm_planes_sizes[i] * 1024 * rotate,
																	(dma_addr_t *)&(osdm->uncommitted_planes[i].dma_addr),
																	GFP_DMA);

			PDEBUG("%s virtual address=0x%8p dma address=0x%8x\n",plane_name_arr[i], osdm->uncommitted_planes[i].vidmem,osdm->uncommitted_planes[i].dma_addr);
			if (!osdm->uncommitted_planes[i].vidmem) {
				dev_err(dev, "Could not allocate %s (%d K-bytes)\n", plane_name_arr[i], dmw96_osdm_planes_sizes[i] * rotate);
				ret = -ENOMEM;
				goto osdm_release_dma_addr;
			}
#ifdef CONFIG_FB_DMW96_OSDM_ROTATE
			osdm->uncommitted_planes[i].rotate_mem = (unsigned int)osdm->uncommitted_planes[i].dma_addr + dmw96_osdm_planes_sizes[i] * 1024;
			PDEBUG("%s rotate_mem = 0x%x\n",plane_name_arr[i], osdm->uncommitted_planes[i].rotate_mem);
#endif
		}
	}

	return 0;

osdm_release_dma_addr:
	for (i=0 ; i < NUM_OF_IN_PLANES ; i++)
		if (dmw96_osdm_planes_sizes[i] != 0)
			if (!osdm->uncommitted_planes[i].vidmem && !osdm->uncommitted_planes[i].dma_addr)
				dma_free_coherent(NULL,
								  dmw96_osdm_planes_sizes[i] * 1024 * rotate,
								  osdm->uncommitted_planes[i].vidmem,
								  osdm->uncommitted_planes[i].info.fix.smem_start);
	return ret;
}

#endif

static int display_ioctl(struct fb_info *info, unsigned int cmd,unsigned long arg)
{
	int retval = 0;
    unsigned int val = 0;
	osdm_rgb888 color = 0;
	struct osdm_regs *regs;
	void __user *argp = (void __user *)arg;
    struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm)
		return -1;

	switch (cmd)
	{
		case FBIOSET_OSDM_DISPLAY_BACKGROUND:
			if (copy_from_user(&color, argp, sizeof(osdm_rgb888)) )
				return -EFAULT;
			osdm->planes[DISPLAY_PLANE].bg_color = color;
			osdm->planes[DISPLAY_PLANE].commited = 1;
			PDEBUG("OSDM Set background red = 0x%4x green = 0x%4x blue = 0x%4x\n", get_red_from_rgb888(color),
				get_green_from_rgb888(color) , get_blue_from_rgb888(color));
			break;
		case FBIOGET_OSDM_DISPLAY_BACKGROUND:
			retval = osdm->planes[DISPLAY_PLANE].bg_color;
			osdm->planes[DISPLAY_PLANE].commited = 1;
			PDEBUG("OSDM Get background red = 0x%4x green = 0x%4x blue = 0x%4x\n", get_red_from_rgb888(color),
				get_green_from_rgb888(color) , get_blue_from_rgb888(color));
			if (copy_to_user(argp, &color , sizeof(osdm_rgb888)) )
				return -EFAULT;
			break;
		case FBIOSET_OSDM_DISPLAY_BACKGROUND_ENB:
			if (copy_from_user(&val, argp, sizeof(unsigned int)) )
				return -EFAULT;
			osdm->planes[DISPLAY_PLANE].bg_enb = val;
			osdm->planes[DISPLAY_PLANE].commited = 1;
			PDEBUG("OSDM Set background is : %s \n", ((val > 0) ? "Enable" : "Dissable"));
			break;
		case FBIOSET_OSDM_DISPLAY_CONFIG:
			/* init_display(); */
			PDEBUG("OSDM display config \n");
			break;
		case FBIOSET_OSDM_DISPLAY_PAUSE:
			if (copy_from_user(&val, argp, sizeof(unsigned int)) )
				return -EFAULT;
			osdm->planes[DISPLAY_PLANE].pause = val; 
			osdm->planes[DISPLAY_PLANE].commited = 1;
			PDEBUG("OSDM is %s\n", (( val > 0 ) ? "Paused" : "Resume from pauseing") );
			break;
		case FBIOSET_OSDM_DISPLAY_LOOP:
			if (copy_from_user(&val, argp, sizeof(unsigned int)) )
				return -EFAULT;
			osdm->planes[DISPLAY_PLANE].loop_mode = val; 
			osdm->planes[DISPLAY_PLANE].commited = 1;
			PDEBUG("OSDM is in loop mode %s \n" , (( val > 0 ) ? "On" : "Off"));
			break;
		case FBIOSET_OSDM_DISPLAY_DITHERING:
			if (copy_from_user(&val, argp, sizeof(unsigned int)) )
				return -EFAULT;
			osdm->planes[DISPLAY_PLANE].dithering = val;
			osdm->planes[DISPLAY_PLANE].commited = 1;
			PDEBUG("OSDM display dithering is %d\n", val);
			break;
		case FBIOSET_OSDM_GET_REGS:
			regs = kmalloc(sizeof(struct osdm_regs), GFP_KERNEL);
			if (!regs)
				return -EFAULT;
			memset(regs, 0 ,sizeof(struct osdm_regs));
			clk_enable(osdm->clk);
			dmw96osdm_get_regs(regs);
			clk_disable(osdm->clk);
			PDEBUG("OSDM display get registers\n");
			retval = copy_to_user(argp, regs , sizeof(struct osdm_regs));
			kfree(regs);
			break;
		default:
			retval = -ENOTTY;
	}

	return retval;
}

static int cfb_osdm_cmap(struct fb_cmap *cmap, struct fb_info *info)
{
	int res = SUCCESS;
    int plane = -1;

	struct dmw96osdm* osdm = FB2OOSDM(info);
	if (!osdm)
		return -1;

	printk("Setting pallete is not supported!\n");
	 return -1;

	if (dmw96_osdm_planes_sizes[MAIN_OSD_PLANE] != 0 || dmw96_osdm_planes_sizes[SUB_OSD_PLANE_1] != 0 || dmw96_osdm_planes_sizes[SUB_OSD_PLANE_2] != 0) {
		/* Pass over all OSD planes */
		plane = get_plane_num(osdm , info);
		if (plane == MAIN_OSD_PLANE || plane == SUB_OSD_PLANE_1 || plane == SUB_OSD_PLANE_2) {
			PDEBUG("OSD palette set\n");
			memcpy((void*)osdm->uncommitted_planes[plane].pal_red,(void*)cmap->red , OSDM_PALETTE_ARRAY_SIZE * 2);
			memcpy((void*)osdm->uncommitted_planes[plane].pal_green,(void*)cmap->green , OSDM_PALETTE_ARRAY_SIZE * 2);
			memcpy((void*)osdm->uncommitted_planes[plane].pal_blue,(void*)cmap->blue , OSDM_PALETTE_ARRAY_SIZE * 2);
			memcpy((void*)osdm->uncommitted_planes[plane].pal_transp,(void*)cmap->transp , OSDM_PALETTE_ARRAY_SIZE * 2);
			osdm->uncommitted_planes[plane].pal_size = cmap->len;
		}
		/*
		for ( i=0 ; i < OSDM_PALETTE_ARRAY_SIZE ; i++)
			PDEBUG("pallete[%d] = 0x%x osdm_type = %d \n" , i , get_rgba8888(osdm->uncommitted_planes[plane].pal_transp[i],
																				 osdm->uncommitted_planes[plane].pal_red[i] ,
																			    osdm->uncommitted_planes[plane].pal_green[i],
		    															osdm->uncommitted_planes[plane].pal_blue[i]), plane );
	   */
        PDEBUG("size %d\n",osdm->uncommitted_planes[plane].pal_size);
	}
	return res;
}

static int cfb_check_var(struct fb_var_screeninfo *var, struct fb_info *info)
{
	int plane_num = -1;
	struct dmw96osdm* osdm = FB2OOSDM(info);
	if (!osdm)
		return -1;

	plane_num = get_plane_num(osdm,info);

	if (plane_num >= 0) {
		if ( (var->rotate != 90) && (var->rotate != 270) && (var->rotate != 0) && (var->rotate != 180) ){
				PDEBUG("%s: invalid rotation angle %d\n",plane_name_arr[plane_num] , var->rotate);
				return -1;
		}
	}
	PDEBUG("\n");
	return 0;
}

const char * ioctl_to_string(unsigned int type)
{
	switch (type)
	{
		case FBIOSET_OSDM_SET_REGS:
			return "FBIOSET_OSDM_SET_REGS";
		case FBIOSET_OSDM_GET_REGS:                    
			return "FBIOSET_OSDM_GET_REGS";
		case FBIOSET_OSDM_DISPLAY_PAUSE:
			return "FBIOSET_OSDM_DISPLAY_PAUSE";
		case FBIOSET_OSDM_DISPLAY_LOOP:
			return "FBIOSET_OSDM_DISPLAY_LOOP";
		case FBIOSET_OSDM_DISPLAY_CONFIG:
			return "FBIOSET_OSDM_DISPLAY_CONFIG";
		case FBIOSET_OSDM_DISPLAY_DITHERING:
			return "FBIOSET_OSDM_DISPLAY_DITHERING";
		case FBIOSET_OSDM_DISPLAY_BACKGROUND:
			return "FBIOSET_OSDM_DISPLAY_BACKGROUND";
		case FBIOGET_OSDM_DISPLAY_BACKGROUND:
			return "FBIOGET_OSDM_DISPLAY_BACKGROUND";
		case FBIOSET_OSDM_DISPLAY_BACKGROUND_ENB:
			return "FBIOSET_OSDM_DISPLAY_BACKGROUND_ENB";
		case FBIOSET_OSDM_PLANE_ENB:
			return "FBIOSET_OSDM_PLANE_ENB";
		case FBIOSET_OSDM_CHROMA_KEY_ENB:
			return "FBIOSET_OSDM_CHROMA_KEY_ENB";
		case FBIOSET_OSDM_PER_PIXEL_ALPHA_ENB:
			return "FBIOSET_OSDM_PER_PIXEL_ALPHA_ENB";
		case FBIOSET_OSDM_CONFIG:
			return "FBIOSET_OSDM_CONFIG";
		case FBIOSET_OSDM_MASK:
			return "FBIOSET_OSDM_MASK";
		case FBIOSET_OSDM_POS:
			return "FBIOSET_OSDM_POS";
		case FBIOGET_OSDM_POS:
			return "FBIOGET_OSDM_POS";
		case FBIOSET_OSDM_TRANSPARENCY_DIM:
			return "FBIOSET_OSDM_TRANSPARENCY_DIM";
		case FBIOGET_OSDM_TRANSPARENCY_DIM:
			return "FBIOGET_OSDM_TRANSPARENCY_DIM";
		case FBIOSET_OSDM_TRANSPARENCY_POS:
			return "FBIOSET_OSDM_TRANSPARENCY_POS";
		case FBIOGET_OSDM_TRANSPARENCY_POS:
			return "FBIOGET_OSDM_TRANSPARENCY_POS";
		case FBIOSET_OSDM_TRANSPARENCY_PIXEL:
			return "FBIOSET_OSDM_TRANSPARENCY_PIXEL";
		case FBIOGET_OSDM_TRANSPARENCY_PIXEL:
			return "FBIOGET_OSDM_TRANSPARENCY_PIXEL";
		case FBIOSET_OSDM_PER_PLANE_ALPHA:
			return "FBIOSET_OSDM_PER_PLANE_ALPHA";
		case FBIOGET_OSDM_PER_PLANE_ALPHA:
			return "FBIOGET_OSDM_PER_PLANE_ALPHA";
		case FBIOSET_OSDM_CHROMA_KEY:
			return "FBIOSET_OSDM_CHROMA_KEY";
		case FBIOGET_OSDM_CHROMA_KEY:
			return "FBIOGET_OSDM_CHROMA_KEY";
		case FBIOSET_OSDM_BYTE_ORDER:
			return "FBIOSET_OSDM_BYTE_ORDER";
		case FBIOGET_OSDM_BYTE_ORDER:
			return "FBIOGET_OSDM_BYTE_ORDER";
		case FBIOSET_OSDM_VIDEO_LIMIT:
			return "FBIOSET_OSDM_VIDEO_LIMIT";
		case FBIOGET_OSDM_VIDEO_LIMIT:
			return "FBIOGET_OSDM_VIDEO_LIMIT";
		case FBIOSET_OSDM_VIDEO_COLOR_CONTROL:
			return "FBIOSET_OSDM_VIDEO_COLOR_CONTROL";
		case FBIOGET_OSDM_VIDEO_COLOR_CONTROL:
			return "FBIOGET_OSDM_VIDEO_COLOR_CONTROL";
		case FBIOSET_OSDM_LUT_BASE:
			return "FBIOSET_OSDM_LUT_BASE";
		case FBIOGET_OSDM_LUT_BASE:
			return "FBIOGET_OSDM_LUT_BASE";
		case FBIOPUT_OSDM_VSCREENINFO:
			return "FBIOPUT_OSDM_VSCREENINFO";
		case FBIOGET_OSDM_VSCREENINFO:
			return "FBIOGET_OSDM_VSCREENINFO";
		case FBIOPAN_OSDM_DISPLAY:
			return "FBIOPAN_OSDM_DISPLAY";
		default:
			return "NONE";
	}
	return "NONE";
}


static int cfb_osdm_pan_display(struct fb_var_screeninfo *var, struct fb_info *info);

static int common_ioctl(struct fb_info *info, unsigned int cmd,unsigned long arg)
{
    int retval = 0;
	void __user *argp = (void __user *)arg;
	struct fb_var_screeninfo dim;
	struct fb_var_screeninfo pan;
	unsigned int val = 0;
    struct osdm_mask mask;
	struct osdm_position pos;
    osdm_abgr8888 t_pixel_color = 0x0;
	unsigned char alpha = 0x0;
	osdm_rgb888 chroma = 0;
	unsigned int byte_order[3] = {0};
	unsigned int lut_base = 0;
	int plane_num = 0;
	unsigned int bpp = 0;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm )
		return -1;

	plane_num = get_plane_num(osdm , info);
	memset (&pos , 0 , sizeof(struct osdm_position));
	memset (&dim , 0 , sizeof(struct fb_var_screeninfo));
	memset (&mask , 0 , sizeof(struct osdm_mask));
	memset (&pan , 0 , sizeof(struct fb_var_screeninfo));

	if (plane_num < 0)
        return -EFAULT;

	PDEBUG("plane=%s ************ S T A R T ********* %s ****************************\n\n",plane_name_arr[plane_num], ioctl_to_string(cmd) );
    switch (cmd) {
		case FBIOSET_OSDM_PLANE_ENB:
            if (copy_from_user(&val, argp, sizeof(unsigned int)) )
				return -EFAULT;
			osdm->uncommitted_planes[plane_num].enb = val;
			PDEBUG("Plane %s is: %s \n",plane_name_arr[plane_num] ,(val > 0) ? "Enable" : "Dissable");
			break;
		case FBIOSET_OSDM_CHROMA_KEY_ENB:
            if (copy_from_user(&val, argp, sizeof(unsigned int)) )
				return -EFAULT;
            osdm->uncommitted_planes[plane_num].chroma_enb = val;
			PDEBUG("Plane %s Chroma key is: %s \n",plane_name_arr[plane_num] , (val > 0) ? "Enable" : "Dissable");
			break;
		case FBIOSET_OSDM_PER_PIXEL_ALPHA_ENB:
            if (copy_from_user(&val, argp, sizeof(unsigned int)) )
				return -EFAULT;
			osdm->uncommitted_planes[plane_num].per_pixel_alpha_enb = val;
            PDEBUG("Plane %s Per pixel alpha is: %s \n",plane_name_arr[plane_num], (val > 0) ? "Enable" : "Dissable");
			break;
		case FBIOSET_OSDM_CONFIG:
            if (copy_from_user(&val, argp, sizeof(unsigned int)) )
				return -EFAULT;
            osdm->uncommitted_planes[plane_num].config = val;
			bpp = dmw96osdm_osdm_get_bpp(osdm->uncommitted_planes[plane_num].config);
			if (bpp != 0) {
				info->var.bits_per_pixel = bpp;
			} else {
				PDEBUG("Invalid config of plane %s 0x%x\n",plane_name_arr[plane_num] , val );
				return -EFAULT;
			}
			dmw96osdm_bitfield_set( &info->var , osdm->uncommitted_planes[plane_num].config);
			PDEBUG("Config of plane %s is 0x%x bpp is %d\n red.length=%d\n red.offset=%d\n green.length=%d\n green.offset=%d\n blue.length=%d\n blue.offset=%d\n transp.length=%d\n transp.offset=%d\n",
					plane_name_arr[plane_num] , val , bpp ,
					info->var.red.length,	
					info->var.red.offset,	
					info->var.green.length,	
					info->var.green.offset,	
					info->var.blue.length,
					info->var.blue.offset,
					info->var.transp.length,	
					info->var.transp.offset);
			break;
		case FBIOSET_OSDM_MASK:
			if (copy_from_user(&mask, argp, sizeof(struct osdm_mask)) )
				return -EFAULT;
            memcpy (&(osdm->uncommitted_planes[plane_num].mask) , &mask , sizeof(struct osdm_mask));
			PDEBUG("Mask of plane %s top=%d bottom=%d right=%d left=%d\n",plane_name_arr[plane_num], mask.top ,  mask.bottom , mask.right , mask.left );
			break;
		case FBIOSET_OSDM_POS:
			if (copy_from_user(&pos, argp, sizeof(struct osdm_position)) )
				return -EFAULT;
			memcpy (&(osdm->uncommitted_planes[plane_num].pos) , &pos , sizeof(struct osdm_position));
			PDEBUG("Position of plane %s is x:%d y:%d \n",plane_name_arr[plane_num] , pos.xpos , pos.ypos);
			break;
		case FBIOSET_OSDM_TRANSPARENCY_DIM:
			if (copy_from_user(&dim, argp, sizeof(struct fb_var_screeninfo)) )
				return -EFAULT;
            memcpy (&(osdm->uncommitted_planes[plane_num].t_dim) , &dim , sizeof(struct fb_var_screeninfo));			
			PDEBUG("Position of transparnt block of plane %s is x:%d y:%d \n",plane_name_arr[plane_num] , dim.xres , dim.yres);
            break;
		case FBIOSET_OSDM_TRANSPARENCY_POS:
			if (copy_from_user(&pos, argp, sizeof(struct osdm_position)) )
				return -EFAULT;
            memcpy (&(osdm->uncommitted_planes[plane_num].t_pos) , &pos , sizeof(struct osdm_position));
			PDEBUG("Dimention of transparency block of plane %s is x:%d y:%d \n",plane_name_arr[plane_num] , pos.xpos , pos.ypos);
            break;
		case FBIOSET_OSDM_TRANSPARENCY_PIXEL:
			if (copy_from_user(&t_pixel_color, argp, sizeof(osdm_abgr8888)) )
				return -EFAULT;
			osdm->uncommitted_planes[plane_num].t_pixel_color = t_pixel_color;
            PDEBUG("Transprency pixel of plane %s is red:%d green:%d blue:%d alpha:%d\n",plane_name_arr[plane_num] ,
					    get_red_from_abgr8888(t_pixel_color),
					    get_green_from_abgr8888(t_pixel_color),
					    get_blue_from_abgr8888(t_pixel_color),
						get_alpha_from_abgr8888(t_pixel_color) );
			break;
		case FBIOSET_OSDM_PER_PLANE_ALPHA:
			if (copy_from_user(&alpha, argp, sizeof(unsigned char)) )
				return -EFAULT;
			osdm->uncommitted_planes[plane_num].per_plane_alpha = alpha;
            PDEBUG("Per plane alpha of plane %s is %d\n",plane_name_arr[plane_num] , alpha );
			break;
		case FBIOSET_OSDM_CHROMA_KEY:
			if (copy_from_user(&chroma, argp, sizeof(osdm_rgb888)) )
				return -EFAULT;
				osdm->uncommitted_planes[plane_num].chroma_key = chroma;
				PDEBUG("Chroma of plane %s is 0x%x\n",plane_name_arr[plane_num] , chroma );
				PDEBUG("Chroma red is %d\n",get_red_from_rgb888(chroma));
				PDEBUG("Chroma green is %d\n",get_green_from_rgb888(chroma));
				PDEBUG("Chroma blue is %d\n",get_blue_from_rgb888(chroma));
			break;
		case FBIOSET_OSDM_BYTE_ORDER:
			if (copy_from_user(byte_order, argp, sizeof(byte_order) ) )
				return -EFAULT;
			memcpy (osdm->uncommitted_planes[plane_num].byte_order, byte_order , sizeof(byte_order));
			PDEBUG("byte order channel 0 of plane %s is 0x%x\n",plane_name_arr[plane_num] , byte_order[0] );
			PDEBUG("byte order channel 1 of plane %s is 0x%x\n",plane_name_arr[plane_num] , byte_order[1] );
			PDEBUG("byte order channel 2 of plane %s is 0x%x\n",plane_name_arr[plane_num] , byte_order[2] );
			break;
		case FBIOSET_OSDM_LUT_BASE:
			if (copy_from_user(&lut_base, argp, sizeof(unsigned int)) )
				return -EFAULT;
				osdm->uncommitted_planes[plane_num].lut_base = lut_base;
				PDEBUG("Lut base of plane %s is 0x%x\n",plane_name_arr[plane_num] , lut_base );
			break;
		case FBIOGET_OSDM_VSCREENINFO:
			if (copy_to_user(argp, &(info->var), sizeof(struct fb_var_screeninfo)) )
				return -EFAULT;
			PDEBUG("size of plane %s is x:%d y:%d \n",plane_name_arr[plane_num] , info->var.xres , info->var.yres);
			PDEBUG("rotate of plane %s is %d \n",plane_name_arr[plane_num] , info->var.rotate);
			PDEBUG("physical address and len of plane %s :0x%lx len:%d \n", plane_name_arr[plane_num], info->fix.smem_start , info->fix.smem_len);
			break;
		case FBIOPUT_OSDM_VSCREENINFO:
			if (copy_from_user(&(info->var), argp, sizeof(struct fb_var_screeninfo)) )
				return -EFAULT;
			info->var.yres_virtual = info->var.yres * 2;
			PDEBUG("size of plane %s is x:%d y:%d \n",plane_name_arr[plane_num] , info->var.xres , info->var.yres);
			PDEBUG("rotate of plane %s is %d \n",plane_name_arr[plane_num] , info->var.rotate);
			PDEBUG("physical address and len of plane %s :0x%lx len:%d \n", plane_name_arr[plane_num], info->fix.smem_start , info->fix.smem_len);
			break;
		case FBIOPAN_OSDM_DISPLAY:
			if (copy_from_user(&(pan), argp, sizeof(struct fb_var_screeninfo)) )
				return -EFAULT;
			retval = cfb_osdm_pan_display(&pan, info);
			PDEBUG("size of plane %s is x:%d y:%d yoffset:%d\n",plane_name_arr[plane_num] , info->var.xres , info->var.yres , pan.yoffset);
			break;
		default:
			retval = display_ioctl(info,cmd,arg);
	}

	PDEBUG("plane=%s ************ END ********* %s **********************\n\n",plane_name_arr[plane_num], ioctl_to_string(cmd) );
	return retval;
}

int cfb_osd_ioctl(struct fb_info *info, unsigned int cmd,unsigned long arg)
{
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm , info);
	int res = 0;

	if (!osdm)
		return -EFAULT;

	down(&osdm->uncommitted_planes[plane_num].setting_sem);
	res = common_ioctl(info, cmd,arg);
	up(&osdm->uncommitted_planes[plane_num].setting_sem);
	
	return res;
}

int video_ioctl(struct fb_info *info, unsigned int cmd,unsigned long arg)
{
	int retval = 0;
	void __user *argp = (void __user *)arg;
	struct osdm_video_color_control color_control;
	int plane_num = 0;
	struct osdm_video_limits video_limits;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm)
		return -1;

	memset (&color_control, 0 , sizeof (struct osdm_video_color_control));
	memset (&video_limits, 0 , sizeof (struct osdm_video_limits));

	plane_num = get_plane_num(osdm , info);
	if (plane_num < 0)
		return -EFAULT;

	switch (cmd) {
		case FBIOSET_OSDM_VIDEO_COLOR_CONTROL:
            if (copy_from_user(&color_control, argp, sizeof(struct osdm_video_color_control)) )
				return -EFAULT;
			memcpy (&(osdm->uncommitted_planes[plane_num].video_color_control) , &color_control , sizeof(struct osdm_video_color_control));
			PDEBUG("Set color control brightness %d\n",osdm->uncommitted_planes[plane_num].video_color_control.brightness);
			PDEBUG("Set color control saturation %d\n",osdm->uncommitted_planes[plane_num].video_color_control.saturation);
			PDEBUG("Set color control hue %d\n",osdm->uncommitted_planes[plane_num].video_color_control.hue);
			PDEBUG("Set color control contrast %d\n",osdm->uncommitted_planes[plane_num].video_color_control.contrast);
            break;
		case FBIOSET_OSDM_VIDEO_LIMIT:
			if (copy_from_user(&video_limits, argp, sizeof(struct osdm_video_limits)) )
				return -EFAULT;
            memcpy (&(osdm->uncommitted_planes[plane_num].video_limits) , &video_limits , sizeof(struct osdm_video_limits));
			PDEBUG("Video limits luma max %d\n", osdm->uncommitted_planes[plane_num].video_limits.luma_max);
			PDEBUG("Video limits luma min %d\n", osdm->uncommitted_planes[plane_num].video_limits.luma_min);
			PDEBUG("Video limits chroma max %d\n", osdm->uncommitted_planes[plane_num].video_limits.chroma_max);
			PDEBUG("Video limits chroma min %d\n", osdm->uncommitted_planes[plane_num].video_limits.chroma_min);
            break;
		default:
			retval = common_ioctl(info, cmd, arg);
	}

	return retval;
}

int cfb_video_ioctl(struct fb_info *info, unsigned int cmd,unsigned long arg)
{
	struct dmw96osdm* osdm = FB2OOSDM(info);
	int retval = 0;
	unsigned int plane_num;

	if (!osdm)
		return -EFAULT;

	plane_num  = get_plane_num(osdm , info);
	if (plane_num < 0)
		return -EFAULT;

	down(&osdm->uncommitted_planes[plane_num].setting_sem);

	retval = video_ioctl(info, cmd, arg);

	up(&osdm->uncommitted_planes[plane_num].setting_sem);
	
	return retval;
}

int cfb_osdm_set_par(struct fb_info *info)
{
	int plane_num = -1;
	struct dmw96osdm* osdm = FB2OOSDM(info);
	if (!osdm)
		return -1;

	plane_num = get_plane_num(osdm,info);
	
	if (plane_num >= 0) {
		PDEBUG("Setting the size of plane %s xres=%d yres=%d\n", plane_name_arr[plane_num],info->var.xres , info->var.yres );
		info->var.yres_virtual = info->var.yres * 2;
	}
    
	return 0;
}
#ifndef CONFIG_MACH_VERSATILE_BROADTILE

int cfb_osdm_mmap(struct fb_info *info, struct vm_area_struct *vma)
{
	int ret = 0;
    int plane_num = -1;
	struct dmw96osdm* osdm = FB2OOSDM(info);

	if (!osdm )
		return -1;

	plane_num = get_plane_num(osdm,info);
	if (plane_num >= 0) {
		ret = dma_mmap_coherent(NULL,
								vma,
								osdm->uncommitted_planes[plane_num].vidmem,
								osdm->uncommitted_planes[plane_num].dma_addr, 
								dmw96_osdm_planes_sizes[plane_num] * 1024);
		if ( ret < 0 ){	
			PDEBUG("Failed to mmap %s\n", plane_name_arr[plane_num]);
		}
	}

    return  ret;
}

#endif

#define PHYS_MEM_OFFSET 0x80000000
int osdm_virt_to_phys(struct mm_struct *mm, unsigned long va,unsigned int* phys_addr)
{
   pgd_t *pgd = pgd_offset(mm, va);
   if (pgd_present(*pgd)) {
      pmd_t *pmd = pmd_offset(pgd, va);
      if (pmd_present(*pmd)) {
		pte_t *pte = pte_offset_map(pmd, va); 
        if (pte_present(*pte)) {
			*phys_addr = (unsigned int)(page_address(pte_page(*pte))) + ( va & ~PAGE_MASK ) - PHYS_MEM_OFFSET;
			PDEBUG("va 0x%lx -> pa 0x%x\n",va, *phys_addr);
			return 0;
		}
     }
   }
   return -1;
}

static int cfb_osdm_pan_display(struct fb_var_screeninfo *var, struct fb_info *info)
{
	int ret = 0;
	int plane_num = -1;
	unsigned int offset = 0;
	struct dmw96osdm* osdm = FB2OOSDM(info);
	
	if (!osdm )
		return -1;
	
	plane_num = get_plane_num(osdm,info);
	
	if ( ( (var->yoffset != info->var.yres) && (var->yoffset != 0) ) || plane_num < 0) {	
		printk("Invalid plane %d or unable to pan plane %d  var->yoffset = %d yoffset should be the size of yres %d or invalid plane num\n",plane_num , plane_num , var->yoffset, info->var.yres );
		return -1;
	}

	PDEBUG("%s  var->yoffset = %d \n", plane_name_arr[plane_num] , var->yoffset );
	offset = (var->yoffset * info->var.xres * info->var.bits_per_pixel) / 8;
	osdm->uncommitted_planes[plane_num].offset = offset;

	ret = dmw96osdm_check_plane_settings(info,plane_num);
	if (FAILED(ret)) {
		printk("Fail testing the setting for plane %s \n", plane_name_arr[plane_num]);
		return ret;
	}
	/* Handle buffer that mapped from userspace */
	osdm->uncommitted_planes[plane_num].info.var.reserved[1] = 0;
	if (var->reserved[0] == PAN_OSDM_ZERO_COPY_MAGIC) {
		unsigned int phys_addr = 0;
		if ( !osdm_virt_to_phys(current->mm , var->reserved[1], &phys_addr) ) {
			osdm->uncommitted_planes[plane_num].info.var.reserved[1] = phys_addr + offset ;
			PDEBUG("OSDM translate virtual address 0x%x to physical 0x%x\n", var->reserved[1] , osdm->uncommitted_planes[plane_num].info.var.reserved[1]);
		}
		else {
			printk("OSDM Failed to translate virtual address 0x%x to physical\n",var->reserved[1]);
			return -1;
		}
	}
	if (info->var.rotate == 90 || info->var.rotate == 180 || info->var.rotate == 270) {
#ifdef CONFIG_FB_DMW96_OSDM_ROTATE
		if (IS_PLANE_OSD(plane_num)) {
			printk("OSDM Failed to rotate OSD - NOT supported for now\n");
			return -1;
		}

		down(&osdm->uncommitted_planes[plane_num].rotate_sem);
		ret = dmw96osdm_plane_rotate_set(&osdm->uncommitted_planes[plane_num].info, plane_num);
		if (SUCCEEDED(ret)) {
			PDEBUG("About to submit DMA list %d to perform rotate\n", (offset ? 1 : 0) );

			ret = dmw96dma_submit(osdm->uncommitted_planes[plane_num].list[ (offset ? 1 : 0) ]);
			if (ret) {
				PDEBUG("FAIL to submit DMA list %d to perform rotate\n", (offset ? 1 : 0) );
				free_rotate_dma_list(plane_num , osdm);
				up(&osdm->uncommitted_planes[plane_num].rotate_sem);	
				return -1;
			}
		} else {
			up(&osdm->uncommitted_planes[plane_num].rotate_sem);	
			return -1;
		}
#else
		printk("OSDM Error, Rotate operation is not supported\n");
		return -1;
#endif		
	} else {
		if (osdm->uncommitted_planes[plane_num].info.var.reserved[1] != 0)
			osdm->uncommitted_planes[plane_num].cur_phy_addr = osdm->uncommitted_planes[plane_num].info.var.reserved[1] + offset ;
		else
			osdm->uncommitted_planes[plane_num].cur_phy_addr = (unsigned int)info->fix.smem_start + offset ;

		if (IS_PLANE_OSD(plane_num))
			down(&osdm->uncommitted_planes[plane_num].pan_sync_sem);

		copy_uncommitted_and_enbale_lcdc_vsync_int(info);
	}
	
	return 0;
}

static int dummy_pan_display(struct fb_var_screeninfo *var, struct fb_info *info){
	return 0;
}

static struct fb_ops dmw96osd_ops = {
	.owner          = THIS_MODULE,
	.fb_check_var	= cfb_check_var,
	.fb_ioctl		= cfb_osd_ioctl,
	.fb_setcmap		= cfb_osdm_cmap,
	.fb_set_par		= cfb_osdm_set_par,
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	.fb_mmap		= cfb_osdm_mmap,
#endif
	.fb_pan_display	= dummy_pan_display,
};
static struct fb_ops dmw96video_ops = {
	.owner          = THIS_MODULE,
	.fb_check_var	= cfb_check_var,
    .fb_ioctl		= cfb_video_ioctl,
	.fb_set_par		= cfb_osdm_set_par,
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	.fb_mmap		= cfb_osdm_mmap,
#endif
	.fb_pan_display	= dummy_pan_display,
};


#define DMW96OSDM_NAME "dmw96osdm"
static char dmw96osdm_name[] = DMW96OSDM_NAME;

/*
 * driver implementation
 */


static int init_background(const struct dmw96osdm *osdm)
{
    dmw96osdm_display_background_set(0x0);
	return SUCCESS;
}


static int 	init_plane(unsigned int osdm_type, struct dmw96osdm *osdm ,struct device *dev)
{
	struct fb_info *info;
    int ret = -ENOMEM;

	info =  &osdm->uncommitted_planes[osdm_type].info;
	if (!info)
		return -ENOMEM;

	info->par = osdm;
	info->flags = FBINFO_FLAG_DEFAULT;
	info->var.nonstd = 0;
	info->screen_base = osdm->uncommitted_planes[osdm_type].vidmem;

	if (osdm_type == MAIN_VIDEO_PLANE || osdm_type == SUB_VIDEO_PLANE)
		info->fbops = &dmw96video_ops;
	else
		info->fbops = &dmw96osd_ops;

    info->flags = FBINFO_FLAG_DEFAULT;
	strcpy(info->fix.id, plane_name_arr[osdm_type]);
	info->fix.type = FB_TYPE_INTERLEAVED_PLANES;
	info->fix.visual = FB_VISUAL_TRUECOLOR;
	info->fix.accel = FB_ACCEL_NONE;
    info->fix.line_length = 0 ;
#ifdef CONFIG_MACH_VERSATILE_BROADTILE
    info->fix.smem_start = osdm->uncommitted_planes[osdm_type].smem_start;
#else
    info->fix.smem_start = (unsigned long)osdm->uncommitted_planes[osdm_type].dma_addr ;
#endif
	info->fix.smem_len = dmw96_osdm_planes_sizes[osdm_type] * 1024;
	info->fix.ypanstep = 1;
	info->screen_size = info->fix.smem_len;
	memset(info->screen_base, 0x00, info->screen_size);

	if (osdm_type == MAIN_OSD_PLANE || osdm_type == SUB_OSD_PLANE_1 || osdm_type == SUB_OSD_PLANE_2) {
		/*allcate the palette with alpha (transparency)*/
		ret = fb_alloc_cmap(&info->cmap, OSDM_PALETTE_ARRAY_SIZE , 1);
		if (ret < 0)
			goto err_free_info_plane;
	}
    /* register framebuffer in userspace */
	ret = register_framebuffer(info);
	if (ret < 0)
		goto err_free_cmap_plane;
	/* Create sysfs entries*/
	ret = create_osdm_sysfs(info);
	if (ret < 0)
		goto err_unregister_fb_plane;
	/* init setting semaphore in case differet proccess access the setting of the same plane */
	sema_init(&osdm->uncommitted_planes[osdm_type].setting_sem,1);
	/* init setting semaphore in case same proccess tries to access twice to DMA */
	sema_init(&osdm->uncommitted_planes[osdm_type].rotate_sem,1);
	/* init semaphore for synchronizing pan operations */
	sema_init(&osdm->uncommitted_planes[osdm_type].pan_sync_sem,1);
	
	PDEBUG("%d\n",osdm_type);

	printk(KERN_INFO "fb%d: %s frame buffer device, %dx%d, %d bpp, %luk\n",
	       info->node,
	       info->fix.id,
	       info->var.xres,
	       info->var.yres,
	       info->var.bits_per_pixel,
	       info->screen_size >> 10);
	
	return 0;

err_unregister_fb_plane:
	unregister_framebuffer(info);
err_free_cmap_plane:
if (osdm_type == MAIN_OSD_PLANE || osdm_type == SUB_OSD_PLANE_1 || osdm_type == SUB_OSD_PLANE_2)
	fb_dealloc_cmap(&info->cmap);
err_free_info_plane:
	framebuffer_release(info);
	return ret;
}

static int init_osdm(struct dmw96osdm *osdm ,struct device *dev )
{
	int ret = 0;

	dmw96osdm_reset_all_plane_settings();
    init_display();
	init_background(osdm);
	dmw96osdm_reset_settings();
	
    if (dmw96_osdm_planes_sizes[MAIN_VIDEO_PLANE] != 0) {
        ret = init_plane(MAIN_VIDEO_PLANE , osdm ,dev);
		if (ret < 0) {
			goto release_osdm6;
		}
	}
	if (dmw96_osdm_planes_sizes[SUB_VIDEO_PLANE] != 0) {
        ret = init_plane(SUB_VIDEO_PLANE , osdm ,dev);
		if (ret < 0) {
			goto release_osdm5;
		}
	}
	if (dmw96_osdm_planes_sizes[MAIN_OSD_PLANE] != 0) {
        ret = init_plane(MAIN_OSD_PLANE , osdm ,dev);
		if (ret < 0) {
			goto release_osdm4;
		}
	}
    if (dmw96_osdm_planes_sizes[SUB_OSD_PLANE_1] != 0) {
        ret = init_plane(SUB_OSD_PLANE_1 , osdm ,dev);
		if (ret < 0) {
			goto release_osdm3;
		}
	}
    if (dmw96_osdm_planes_sizes[SUB_OSD_PLANE_2] != 0) {
        ret = init_plane(SUB_OSD_PLANE_2 , osdm ,dev);
		if (ret < 0) {
			goto release_osdm2;
		}
	}
	return 0;
release_osdm2:
if (dmw96_osdm_planes_sizes[SUB_OSD_PLANE_1] != 0)
	framebuffer_release(&osdm->uncommitted_planes[SUB_OSD_PLANE_1].info);
release_osdm3:
if (dmw96_osdm_planes_sizes[MAIN_OSD_PLANE] != 0)
	framebuffer_release(&osdm->uncommitted_planes[MAIN_OSD_PLANE].info);
release_osdm4:
if (dmw96_osdm_planes_sizes[SUB_VIDEO_PLANE] != 0)
	framebuffer_release(&osdm->uncommitted_planes[SUB_VIDEO_PLANE].info);
release_osdm5:
if (dmw96_osdm_planes_sizes[MAIN_VIDEO_PLANE] != 0)
	framebuffer_release(&osdm->uncommitted_planes[MAIN_VIDEO_PLANE].info);
release_osdm6:
	return ret;
}

static irqreturn_t dmw96osdm_irq(int irq, void *priv)
{
    //struct dmw96osdm *osdm = priv;
	unsigned int i;
	PDEBUG_START
	if ( dmw96osdm_is_display_finished() ) {
		dmw96osdm_mix_pre_settings();
		dmw96osdm_interrupt_handle();
		g_osdm->is_working = 0;
		pan_osdm();
		/* if there is job which was commited then enable LCDC int*/
		if (g_osdm->planes[MAIN_VIDEO_PLANE].commited ||
			g_osdm->planes[SUB_VIDEO_PLANE].commited ||
			g_osdm->planes[MAIN_OSD_PLANE].commited ||
			g_osdm->planes[SUB_OSD_PLANE_1].commited ||
			g_osdm->planes[SUB_OSD_PLANE_2].commited ) {
			PDEBUG("There are commited left enable Vsync interrupt\n");
			lcdc_enable_vsync_int_once();
		}

		for (i = MAIN_OSD_PLANE ; i <= SUB_OSD_PLANE_2 ; i++ )
			if (g_osdm->planes[i].just_committed) {
				 g_osdm->planes[i].just_committed = 0;
				 up(&g_osdm->uncommitted_planes[i].pan_sync_sem);
			}
		
		clk_disable(g_osdm->clk);
	} else
		printk("Error DMA OSDM not finished");
	PDEBUG_END
	return IRQ_HANDLED;
}

static int __init dmw96osdm_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
    struct dmw96osdm *dmw96osdm;
    struct resource *res;
	int ret = 0;
#ifdef CONFIG_MACH_VERSATILE_BROADTILE
	int intr_type = IRQF_SHARED;
#else
	int intr_type = 0;
#endif
    PDEBUG("%s()\n", __FUNCTION__);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "need a memory resource\n");
		return -ENXIO;
	}

	if (!request_mem_region(res->start, res->end - res->start,
	                        "dmw96osdm regs")) {
		dev_err(dev, "failed to request memory region\n");
		return -EBUSY;
	}

    dmw96osdm = kzalloc(sizeof(*dmw96osdm), GFP_KERNEL);
	if (!dmw96osdm)
		return -ENOMEM;

	g_osdm = dmw96osdm;

	dmw96osdm->irq = platform_get_irq(pdev, 0);
	if (dmw96osdm->irq < 0) {
		dev_err(&pdev->dev, "need an irq resource\n");
		ret = dmw96osdm->irq;
		goto err_free_dmw96osdm;
	}
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	ret = request_irq(dmw96osdm->irq, dmw96osdm_irq, intr_type , dmw96osdm_name, dmw96osdm);
	if (ret < 0) {
		dev_err(dev, "request irq %d failed\n", dmw96osdm->irq);
		goto err_free_dmw96osdm;
	}
#endif
    /* init hardware */
	osdm_regs = ioremap(res->start, res->end - res->start);
	if (!osdm_regs) {
		dev_err(dev, "unable to map registers\n");
		ret = -ENOMEM;
		goto err_free_dmw96osdm;
	}
	
    PDEBUG("ioremap osdm registres %p\n", osdm_regs);

    ret = allocate_osdm_buffers(dmw96osdm ,dev);
    if (ret < 0) {
		dev_err(dev, "unable to allocate osdm buffers\n");
		goto err_free_dmw96osdm;
	}

#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	dmw96osdm->clk = clk_get(&pdev->dev, NULL);
	if (IS_ERR(dmw96osdm->clk)) {
		ret = PTR_ERR(dmw96osdm->clk);
		goto err_free_dmw96osdm;
	}
	clk_enable(dmw96osdm->clk);
#endif
    /*init osdm*/
	ret = -ENOMEM;
	ret = init_osdm(dmw96osdm , dev);
	if (ret < 0) {
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
		clk_disable(dmw96osdm->clk);
#endif
		dev_err(dev, "unable to init osdm\n");
		goto err_free_dmw96osdm;
	}
	/* attach driver data to the device */
    dev_set_drvdata(dev, dmw96osdm);
	dmw96osdm_interrupt_clear(0xffffffff);
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	clk_disable(dmw96osdm->clk);
#endif
	spin_lock_init(dmw96osdm->copy_lock);
	dmw96osdm->is_working = 0;

	notifier.notifier_call = dmw96osdm_notifier_callback;
	fb_register_client(&notifier);

	printk("success to set the osdm to driver\n");
	return 0;

err_free_dmw96osdm:
	kfree(dmw96osdm);
	g_osdm = NULL;
	return ret;
}

static int __exit dmw96osdm_remove(struct platform_device *pdev)
{
	/* TODO: free everything */
	g_osdm = NULL;
	return 0;
}

static struct platform_driver dmw96osdm_dirver = {
	.remove = __exit_p(dmw96osdm_remove),
	.driver = {
		.name = dmw96osdm_name,
		.owner = THIS_MODULE,
	},
};

static int parse_size(char *wp,unsigned int plane)
{
	 char *s;

	 if ((s = strsep(&wp, ":")) == NULL)
			 return -EINVAL;
	 dmw96_osdm_planes_sizes[plane] = simple_strtoul(s, NULL, 0);
	 PDEBUG("dmw96_osdm_planes_sizes[%d] = %d\n", plane, dmw96_osdm_planes_sizes[plane]  );
	 return 0;
}
#ifndef MODULE
/* Process kernel command line parameters */
static int __init dmw96osdm_setup(char *options)
{
    char *this_opt = NULL;

	PDEBUG("dmw96osdm: options %s\n", options);
	if (!options || !*options)
		return 0;
    
    while ((this_opt = strsep(&options, ":")) != NULL) {
		 if (!*this_opt)
			 continue;
		 if (!strncmp(this_opt, "vid0=", 5)){
			 if (!strncmp(this_opt + 5, "off", 3))
				 dmw96_osdm_planes_sizes[MAIN_VIDEO_PLANE] = 0;
			 else if (parse_size(this_opt + 5,MAIN_VIDEO_PLANE))
				 printk("OSDM no vid0 parameter\n");
		 }
		 if (!strncmp(this_opt, "vid1=", 5)) {
			 if (!strncmp(this_opt + 5, "off", 3))
				 dmw96_osdm_planes_sizes[SUB_VIDEO_PLANE] = 0;
			 else if (parse_size(this_opt + 5,SUB_VIDEO_PLANE))
				 printk("OSDM no vid1 parameter\n");
		 }
		 if (!strncmp(this_opt, "osd0=", 5)) {
			 if (!strncmp(this_opt + 5, "off", 3))
				 dmw96_osdm_planes_sizes[MAIN_OSD_PLANE] = 0;
			 else if (parse_size(this_opt + 5,MAIN_OSD_PLANE))
				 printk("OSDM no osd0 parameter\n");
		 }
		 if (!strncmp(this_opt, "osd1=", 5)) {
			 if (!strncmp(this_opt + 5, "off", 3))
				 dmw96_osdm_planes_sizes[SUB_OSD_PLANE_1] = 0;
			 else if (parse_size(this_opt + 5,SUB_OSD_PLANE_1))
				 printk("OSDM no osd1 parameter\n");
		 }
		 if (!strncmp(this_opt, "osd2=", 5)) {
			 if (!strncmp(this_opt + 5, "off", 3))
				 dmw96_osdm_planes_sizes[SUB_OSD_PLANE_2] = 0;
			 else if (parse_size(this_opt + 5,SUB_OSD_PLANE_2))
				 printk("OSDM no osd2 parameter\n");
		 }
	}
	return 0;
}

#endif

static int __init dmw96osdm_init(void)
{
#ifndef MODULE
	char *option;

	if (fb_get_options("dmw96osdmfb", &option)) {
		printk("Fail getting options for dmw96osdm\n");
		return -ENODEV;
	}
	else if (option) {
		PDEBUG("Success getting options for dmw96osdm\n");
		dmw96osdm_setup(option);
	}
#endif
    return platform_driver_probe(&dmw96osdm_dirver, dmw96osdm_probe);
}

static void __exit dmw96osdm_exit(void)
{
	platform_driver_unregister(&dmw96osdm_dirver);
}

module_init(dmw96osdm_init);
module_exit(dmw96osdm_exit);
