/***************************************************************************
*
* SIMG PART NUMBER - HDMI Transmitter Driver
*
* Copyright (C) (2011, Silicon Image)
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation version 2.
*
* This program is distributed "as is" WITHOUT ANY WARRANTY of any
* kind, whether express or implied; without even the implied warranty
* of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
*****************************************************************************/

#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/irq.h>
#include <linux/debugfs.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/console.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <mach/hdmi.h>
#include <linux/switch.h>
#include <linux/notifier.h>
#include <linux/fb.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>

#include "siHdmiTx_902x_TPI.h"


/* Sound : ALSA interface: */
/* ----------------------  */

//#define DEBUG

#ifdef DEBUG
	#define dbg_prt(fmt, ...) \
		printk(KERN_DEBUG "HDMI SI Codec-drv: %s(): " fmt "\n", __func__, ##__VA_ARGS__)

	#define DIR_STR(stream_dir)  ((stream_dir == 0) ? "for Playback" : "for Capture")
#else
	#define dbg_prt(fmt, ...)
	#define DIR_SRT(dir)
#endif

#define DEVICE_NAME	"sii902xA"
#define HDMI_FORMATS	(SNDRV_PCM_FMTBIT_S16_LE)
#define HDMI_RATES	(SNDRV_PCM_RATE_32000	|	\
			 SNDRV_PCM_RATE_44100	|	\
			 SNDRV_PCM_RATE_48000)

/******************************************************************************/

static struct notifier_block notifier;

struct hdmi_switch_data {
	struct switch_dev sdev;
	unsigned gpio;
	unsigned int status;
	struct work_struct work;
	struct attribute_group *attr_group;
};
static struct hdmi_switch_data *switch_data;

static struct i2c_client *sii902xA = NULL;

enum {
	LCD_DISPLAY_UNKNOWN = -1,
	LCD_DISPLAY_LCD,
	LCD_DISPLAY_HDMI,
};

static struct {
	unsigned char HDMIVideoFormat;	// 0 = CEA-861 VIC; 1 = HDMI_VIC; 2 = 3D
	unsigned char VIC;		// VIC or the HDMI_VIC
	unsigned char AspectRatio;	// 4x3 or 16x9
	unsigned char ColorSpace;	// 0 = RGB; 1 = YCbCr4:4:4; 2 = YCbCr4:2:2_16bits; 3 = YCbCr4:2:2_8bits; 4 = xvYCC4:4:4
	unsigned char ColorDepth;	// 0 = 8bits; 1 = 10bits; 2 = 12bits
	unsigned char Colorimetry;	// 0 = 601; 1 = 709
	unsigned char SyncMode;		// 0 = external HS/VS/DE; 1 = external HS/VS and internal DE; 2 = embedded sync
	unsigned char TclkSel;		// 0 = x0.5CLK; 1 = x1CLK; 2 = x2CLK; 3 = x4CLK
	unsigned char ThreeDStructure;	// Valid when (HDMIVideoFormat == VMD_HDMIFORMAT_3D)
	unsigned char ThreeDExtData;	// Valid when (HDMIVideoFormat == VMD_HDMIFORMAT_3D) && (ThreeDStructure == VMD_3D_SIDEBYSIDEHALF)

	/* Sound: 
	 * Note: all(!) values are bit-coded (not natural numbers)
	 */
	unsigned char AudioMode;	// I2S, S/PDIF, HBR, DSD
	unsigned char AudioInFs;	// Sample-Rate on input TDM bus	
	unsigned char AudioOutFs;	// Sample-Rate on output HDMI bus
	unsigned char AudioInWrdLen;	// 16, 20 or 24 bits on input TDM bus
	unsigned char AudioOutWrdLen;	// 16 to 24 bits on output HDMI bus
	unsigned char AudioI2sDataPort;	// SD0, SD1, SD2, SD3 TDM inputs
	unsigned char AudioI2SFormat;	// TDM format details (see TPI reg 0x20)
	/* ================================================================= */
	
	unsigned char txPowerState;
	unsigned char hdmiCableConnected;
	unsigned char dsRxPoweredUp;
	unsigned char tpivmode[3];  // saved TPI Reg0x08/Reg0x09/Reg0x0A values.
	int lcd_display_mode;
} siHdmiTx;

static unsigned char
ReadByteTPI(int reg)
{
	return i2c_smbus_read_byte_data(sii902xA, reg);
}

static void
WriteByteTPI(int reg, unsigned char data)
{
	i2c_smbus_write_byte_data(sii902xA, reg, data);
}

static void
ReadSetWriteTPI(unsigned char Offset, unsigned char Pattern)
{
	unsigned char tmp;

	tmp = ReadByteTPI(Offset);
	tmp |= Pattern;
	WriteByteTPI(Offset, tmp);
}

static void
ReadClearWriteTPI(unsigned char Offset, unsigned char Pattern)
{
	unsigned char tmp;

	tmp = ReadByteTPI(Offset);
	tmp &= ~Pattern;
	WriteByteTPI(Offset, tmp);
}

static void
ReadModifyWriteTPI(unsigned char Offset, unsigned char Mask, unsigned char Value)
{
	unsigned char tmp;

	tmp = ReadByteTPI(Offset);
	tmp &= ~Mask;
	tmp |= (Value & Mask);
	WriteByteTPI(Offset, tmp);
}

static void
WriteBlockTPI(unsigned char TPI_Offset, unsigned short NBytes, unsigned char * pData)
{
	i2c_smbus_write_i2c_block_data(sii902xA,TPI_Offset, NBytes, pData);
}

static void
WriteIndexedRegister(unsigned char PageNum, unsigned char RegOffset, unsigned char RegValue)
{
	WriteByteTPI(TPI_INTERNAL_PAGE_REG, PageNum);    // Internal page
	WriteByteTPI(TPI_INDEXED_OFFSET_REG, RegOffset); // Indexed register
	WriteByteTPI(TPI_INDEXED_VALUE_REG, RegValue);   // Read value into buffer
}

static void
ReadModifyWriteIndexedRegister(unsigned char PageNum, unsigned char RegOffset,
				unsigned char Mask, unsigned char Value)
{
	unsigned char tmp;

	WriteByteTPI(TPI_INTERNAL_PAGE_REG, PageNum);
	WriteByteTPI(TPI_INDEXED_OFFSET_REG, RegOffset);

	tmp = ReadByteTPI(TPI_INDEXED_VALUE_REG);
	tmp &= ~Mask;
	tmp |= (Value & Mask);
	WriteByteTPI(TPI_INDEXED_VALUE_REG, tmp);
}

static void
TxHW_Reset(void)
{
	struct hdmi_platform_data *Sii902xA_pdata;

	TPI_TRACE_PRINT(("HDMI: TxHW_Reset()\n"));

	Sii902xA_pdata = sii902xA->dev.platform_data;

	if (Sii902xA_pdata->reset)
		Sii902xA_pdata->reset();
}

static void
EnableTMDS(void)
{
	TPI_DEBUG_PRINT(("TMDS -> Enabled\n"));
	ReadModifyWriteTPI(TPI_SYSTEM_CONTROL_DATA_REG,
	                   TMDS_OUTPUT_CONTROL_MASK | AV_MUTE_MASK,
	                   TMDS_OUTPUT_CONTROL_ACTIVE | AV_MUTE_NORMAL);

	WriteByteTPI(TPI_PIX_REPETITION, siHdmiTx.tpivmode[0]);
}

static void
DisableTMDS(void)
{
	TPI_DEBUG_PRINT(("TMDS -> Disabled\n"));
	ReadModifyWriteTPI(TPI_SYSTEM_CONTROL_DATA_REG,
	                   TMDS_OUTPUT_CONTROL_MASK | AV_MUTE_MASK,
	                   TMDS_OUTPUT_CONTROL_POWER_DOWN | AV_MUTE_MUTED);
}

static unsigned char
EnableInterrupts(unsigned char Interrupt_Pattern)
{
	TPI_TRACE_PRINT(("HDMI: EnableInterrupts()\n"));
	ReadSetWriteTPI(TPI_INTERRUPT_ENABLE_REG, Interrupt_Pattern);
	return TRUE;
}

/***
static unsigned char
DisableInterrupts(unsigned char Interrupt_Pattern)
{
	TPI_TRACE_PRINT(("HDMI: DisableInterrupts()\n"));
	ReadClearWriteTPI(TPI_INTERRUPT_ENABLE_REG, Interrupt_Pattern);

	return TRUE;
}
***/

static void
siHdmiTx_PowerStateD0(void)
{
	ReadModifyWriteTPI(TPI_DEVICE_POWER_STATE_CTRL_REG,
				TX_POWER_STATE_MASK, TX_POWER_STATE_D0);
	TPI_DEBUG_PRINT(("TX Power State D0\n"));
	siHdmiTx.txPowerState = TX_POWER_STATE_D0;
}

static void
siHdmiTx_PowerStateD2(void)
{
	ReadModifyWriteTPI(TPI_DEVICE_POWER_STATE_CTRL_REG,
				TX_POWER_STATE_MASK, TX_POWER_STATE_D2);
	TPI_DEBUG_PRINT(("TX Power State D2\n"));
	siHdmiTx.txPowerState = TX_POWER_STATE_D2;
}

//------------------------------------------------------------------------------
// Video mode table
//------------------------------------------------------------------------------
struct ModeIdType {
	unsigned char Mode_C1;
	unsigned char Mode_C2;
	unsigned char SubMode;
};

struct PxlLnTotalType {
	unsigned short Pixels;
	unsigned short Lines;
};

struct HVPositionType {
	unsigned short H;
	unsigned short V;
};

struct HVResolutionType {
	unsigned short H;
	unsigned short V;
};

struct TagType {
	unsigned char RefrTypeVHPol;
	unsigned short VFreq;
	struct PxlLnTotalType Total;
};

struct _656Type {
	unsigned char IntAdjMode;
	unsigned short HLength;
	unsigned char VLength;
	unsigned short Top;
	unsigned short Dly;
	unsigned short HBit2HSync;
	unsigned char VBit2VSync;
	unsigned short Field2Offset;
};

struct Vspace_Vblank {
	unsigned char VactSpace1;
	unsigned char VactSpace2;
	unsigned char Vblank1;
	unsigned char Vblank2;
	unsigned char Vblank3;
};

struct VModeInfoType {
	struct ModeIdType ModeId;
	unsigned int PixClk;
	struct TagType Tag;
	struct HVPositionType Pos;
	struct HVResolutionType Res;
	unsigned char AspectRatio;
	struct _656Type _656;
	unsigned char PixRep;
	struct Vspace_Vblank VsVb;
	unsigned char _3D_Struct;
};

#define NSM                     0   // No Sub-Mode

#define DEFAULT_VIDEO_MODE	0

#define ProgrVNegHNeg           0x00
#define ProgrVNegHPos           0x01
#define ProgrVPosHNeg           0x02
#define ProgrVPosHPos           0x03

#define InterlaceVNegHNeg	0x04
#define InterlaceVPosHNeg	0x05
#define InterlaceVNgeHPos	0x06
#define InterlaceVPosHPos	0x07

#define VIC_BASE                0
#define HDMI_VIC_BASE           43

// Aspect ratio
#define R_4	0   // 4:3
#define R_4or16	1   // 4:3 or 16:9
#define R_16	2   // 16:9

static struct VModeInfoType VModesTable[] =
{
    //===================================================================================================================================================================================================================================
    //         VIC                  Refresh type Refresh-Rate Pixel-Totals  Position     Active     Aspect   Int  Length          Hbit  Vbit  Field  Pixel          Vact Space/Blank
    //        1   2  SubM   PixClk  V/H Position       VFreq   H      V      H    V       H    V    Ratio    Adj  H   V  Top  Dly HSync VSync Offset Repl  Space1 Space2 Blank1 Blank2 Blank3  3D
    //===================================================================================================================================================================================================================================
    {{        1,  0, NSM},  2517,  {ProgrVNegHNeg,     6000, { 800,  525}}, {144, 35}, { 640, 480}, R_4,     {0,  96, 2, 33,  48,  16,  10,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 0 - 1.       640  x 480p @ 60 VGA
    {{        2,  3, NSM},  2700,  {ProgrVNegHNeg,     6000, { 858,  525}}, {122, 36}, { 720, 480}, R_4or16, {0,  62, 6, 30,  60,  19,   9,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 1 - 2,3      720  x 480p
    {{        4,  0, NSM},  7425,  {ProgrVPosHPos,     6000, {1650,  750}}, {260, 25}, {1280, 720}, R_16,    {0,  40, 5, 20, 220, 110,   5,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 2 - 4        1280 x 720p@60Hz
    {{        5,  0, NSM},  7425,  {InterlaceVPosHPos, 6000, {2200,  562}}, {192, 20}, {1920,1080}, R_16,    {0,  44, 5, 15, 148,  88,   2, 1100},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 3 - 5        1920 x 1080i
    {{        6,  7, NSM},  2700,  {InterlaceVNegHNeg, 6000, {1716,  264}}, {119, 18}, { 720, 480}, R_4or16, {3,  62, 3, 15, 114,  17,   5,  429},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 4 - 6,7      1440 x 480i,pix repl
    {{        8,  9,   1},  2700,  {ProgrVNegHNeg,     6000, {1716,  262}}, {119, 18}, {1440, 240}, R_4or16, {0, 124, 3, 15, 114,  38,   4,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 5 - 8,9(1)   1440 x 240p
    {{        8,  9,   2},  2700,  {ProgrVNegHNeg,     6000, {1716,  263}}, {119, 18}, {1440, 240}, R_4or16, {0, 124, 3, 15, 114,  38,   4,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 6 - 8,9(2)   1440 x 240p
    {{       10, 11, NSM},  5400,  {InterlaceVNegHNeg, 6000, {3432,  525}}, {238, 18}, {2880, 480}, R_4or16, {0, 248, 3, 15, 228,  76,   4, 1716},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 7 - 10,11    2880 x 480i
    {{       12, 13,   1},  5400,  {ProgrVNegHNeg,     6000, {3432,  262}}, {238, 18}, {2880, 240}, R_4or16, {0, 248, 3, 15, 228,  76,   4,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 8 - 12,13(1) 2880 x 240p
    {{       12, 13,   2},  5400,  {ProgrVNegHNeg,     6000, {3432,  263}}, {238, 18}, {2880, 240}, R_4or16, {0, 248, 3, 15, 228,  76,   4,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 9 - 12,13(2) 2880 x 240p
    {{       14, 15, NSM},  5400,  {ProgrVNegHNeg,     6000, {1716,  525}}, {244, 36}, {1440, 480}, R_4or16, {0, 124, 6, 30, 120,  32,   9,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 10 - 14,15    1440 x 480p
    {{       16,  0, NSM}, 14835,  {ProgrVPosHPos,     6000, {2200, 1125}}, {192, 41}, {1920,1080}, R_16,    {0,  44, 5, 36, 148,  88,   4,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 11 - 16       1920 x 1080p
    {{       17, 18, NSM},  2700,  {ProgrVNegHNeg,     5000, { 864,  625}}, {132, 44}, { 720, 576}, R_4or16, {0,  64, 5, 39,  68,  12,   5,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 12 - 17,18    720  x 576p
    {{       19,  0, NSM},  7425,  {ProgrVPosHPos,     5000, {1980,  750}}, {260, 25}, {1280, 720}, R_16,    {0,  40, 5, 20, 220, 440,   5,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 13 - 19       1280 x 720p@50Hz
    {{       20,  0, NSM},  7425,  {InterlaceVPosHPos, 5000, {2640, 1125}}, {192, 20}, {1920,1080}, R_16,    {0,  44, 5, 15, 148, 528,   2, 1320},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 14 - 20       1920 x 1080i
    {{       21, 22, NSM},  2700,  {InterlaceVNegHNeg, 5000, {1728,  625}}, {132, 22}, { 720, 576}, R_4,     {3,  63, 3, 19, 138,  24,   2,  432},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 15 - 21,22    1440 x 576i
    {{       23, 24,   1},  2700,  {ProgrVNegHNeg,     5000, {1728,  312}}, {132, 22}, {1440, 288}, R_4or16, {0, 126, 3, 19, 138,  24,   2,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 16 - 23,24(1) 1440 x 288p
    {{       23, 24,   2},  2700,  {ProgrVNegHNeg,     5000, {1728,  313}}, {132, 22}, {1440, 288}, R_4or16, {0, 126, 3, 19, 138,  24,   2,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 17 - 23,24(2) 1440 x 288p
    {{       23, 24,   3},  2700,  {ProgrVNegHNeg,     5000, {1728,  314}}, {132, 22}, {1440, 288}, R_4or16, {0, 126, 3, 19, 138,  24,   2,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 18 - 23,24(3) 1440 x 288p
    {{       25, 26, NSM},  5400,  {InterlaceVNegHNeg, 5000, {3456,  625}}, {264, 22}, {2880, 576}, R_4or16, {0, 252, 3, 19, 276,  48,   2, 1728},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 19 - 25,26    2880 x 576i
    {{       27, 28,   1},  5400,  {ProgrVNegHNeg,     5000, {3456,  312}}, {264, 22}, {2880, 288}, R_4or16, {0, 252, 3, 19, 276,  48,   2,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 20 - 27,28(1) 2880 x 288p
    {{       27, 28,   2},  5400,  {ProgrVNegHNeg,     5000, {3456,  313}}, {264, 22}, {2880, 288}, R_4or16, {0, 252, 3, 19, 276,  48,   3,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 21 - 27,28(2) 2880 x 288p
    {{       27, 28,   3},  5400,  {ProgrVNegHNeg,     5000, {3456,  314}}, {264, 22}, {2880, 288}, R_4or16, {0, 252, 3, 19, 276,  48,   4,    0},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 22 - 27,28(3) 2880 x 288p
    {{       29, 30, NSM},  5400,  {ProgrVPosHNeg,     5000, {1728,  625}}, {264, 44}, {1440, 576}, R_4or16, {0, 128, 5, 39, 136,  24,   5,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 23 - 29,30    1440 x 576p
    {{       31,  0, NSM}, 14850,  {ProgrVPosHPos,     5000, {2640, 1125}}, {192, 41}, {1920,1080}, R_16,    {0,  44, 5, 36, 148, 528,   4,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 24 - 31(1)    1920 x 1080p
    {{/*7417*/32, 0, NSM},  7425,  {ProgrVPosHPos,     2400, {2750, 1125}}, {192, 41}, {1920,1080}, R_16,    {0,  44, 5, 36, 148, 638,   4,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 25 - 32(2)    1920 x 1080p@24Hz
    {{       33,  0, NSM},  7425,  {ProgrVPosHPos,     2500, {2640, 1125}}, {192, 41}, {1920,1080}, R_16,    {0,  44, 5, 36, 148, 528,   4,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 26 - 33(3)    1920 x 1080p
    {{/*7417*/34, 0, NSM},  7425,  {ProgrVPosHPos,     3000, {2200, 1125}}, {192, 41}, {1920,1080}, R_16,    {0,  44, 5, 36, 148, 528,   4,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 27 - 34(4)    1920 x 1080p
    {{       35, 36, NSM}, 10800,  {ProgrVNegHNeg,     5994, {3432,  525}}, {488, 36}, {2880, 480}, R_4or16, {0, 248, 6, 30, 240,  64,  10,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 28 - 35, 36   2880 x 480p@59.94/60Hz
    {{       37, 38, NSM}, 10800,  {ProgrVNegHNeg,     5000, {3456,  625}}, {272, 39}, {2880, 576}, R_4or16, {0, 256, 5, 40, 272,  48,   5,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 29 - 37, 38   2880 x 576p@50Hz
    {{       39,  0, NSM},  7200,  {InterlaceVNegHNeg, 5000, {2304, 1250}}, {352, 62}, {1920,1080}, R_16,    {0, 168, 5, 87, 184,  32,  24,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 30 - 39       1920 x 1080i@50Hz
    {{       40,  0, NSM}, 14850,  {InterlaceVPosHPos, 10000,{2640, 1125}}, {192, 20}, {1920,1080}, R_16,    {0,  44, 5, 15, 148, 528,   2, 1320},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 31 - 40       1920 x 1080i@100Hz
    {{       41,  0, NSM}, 14850,  {InterlaceVPosHPos, 10000,{1980,  750}}, {260, 25}, {1280, 720}, R_16,    {0,  40, 5, 20, 220, 400,   5,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 32 - 41       1280 x 720p@100Hz
    {{       42, 43, NSM},  5400,  {ProgrVNegHNeg,     10000,{ 864,  144}}, {132, 44}, { 720, 576}, R_4or16, {0,  64, 5, 39,  68,  12,   5,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 33 - 42, 43,  720p x 576p@100Hz
    {{       44, 45, NSM},  5400,  {InterlaceVNegHNeg, 10000,{ 864,  625}}, {132, 22}, { 720, 576}, R_4or16, {0,  63, 3, 19,  69,  12,   2,  432},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 34 - 44, 45,  720p x 576i@100Hz, pix repl
    {{       46,  0, NSM}, 14835,  {InterlaceVPosHPos, 11988,{2200, 1125}}, {192, 20}, {1920,1080}, R_16,    {0,  44, 5, 15, 149,  88,   2, 1100},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 35 - 46,      1920 x 1080i@119.88/120Hz
    {{       47,  0, NSM}, 14835,  {ProgrVPosHPos,     11988,{1650,  750}}, {260, 25}, {1280, 720}, R_16,    {0,  40, 5, 20, 220, 110,   5, 1100},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 36 - 47,      1280 x 720p@119.88/120Hz
    {{       48, 49, NSM},  5400,  {ProgrVNegHNeg,     11988,{ 858,  525}}, {122, 36}, { 720, 480}, R_4or16, {0,  62, 6, 30,  60,  16,  10,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 37 - 48, 49   720  x 480p@119.88/120Hz
    {{       50, 51, NSM},  5400,  {InterlaceVNegHNeg, 11988,{ 858,  525}}, {119, 18}, { 720, 480}, R_4or16, {0,  62, 3, 15,  57,  19,   4,  429},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 38 - 50, 51   720  x 480i@119.88/120Hz
    {{       52, 53, NSM}, 10800,  {ProgrVNegHNeg,     20000,{ 864,  625}}, {132, 44}, { 720, 576}, R_4or16, {0,  64, 5, 39,  68,  12,   5,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 39 - 52, 53,  720  x 576p@200Hz
    {{       54, 55, NSM}, 10800,  {InterlaceVNegHNeg, 20000,{ 864,  625}}, {132, 22}, { 720, 576}, R_4or16, {0,  63, 3, 19,  69,  12,   2,  432},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 40 - 54, 55,  1440 x 576i @200Hz, pix repl
    {{       56, 57, NSM}, 10800,  {ProgrVNegHNeg,     24000,{ 858,  525}}, {122, 42}, { 720, 480}, R_4or16, {0,  62, 6, 30,  60,  16,   9,    0},    0,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 41 - 56, 57,  720  x 480p @239.76/240Hz
    {{       58, 59, NSM}, 10800,  {InterlaceVNegHNeg, 24000,{ 858,  525}}, {119, 18}, { 720, 480}, R_4or16, {0,  62, 3, 15,  57,  19,   4,  429},    1,    {0,     0,     0,     0,    0},    NO_3D_SUPPORT}, // 42 - 58, 59,  1440 x 480i @239.76/240Hz, pix repl
};


//------------------------------------------------------------------------------
// Aspect Ratio table defines the aspect ratio as function of VIC. This table
// should be used in conjunction with the 861-D part of VModeInfoType VModesTable[]
// (formats 0 - 59) because some formats that differ only in their AR are grouped
// together (e.g., formats 2 and 3).
//------------------------------------------------------------------------------
static unsigned char AspectRatioTable[] = {
	R_4,  R_4,  R_16, R_16, R_16, R_4,  R_16, R_4,  R_16, R_4,
	R_16, R_4,  R_16, R_4,  R_16, R_16, R_4,  R_16, R_16, R_16,
	R_4,  R_16, R_4,  R_16, R_4,  R_16, R_4,  R_16, R_4,  R_16,
	R_16, R_16, R_16, R_16, R_4,  R_16, R_4,  R_16, R_16, R_16,
	R_16, R_4,  R_16, R_4,  R_16, R_16, R_16, R_4,  R_16, R_4,
	R_16, R_4,  R_16, R_4,  R_16, R_4,  R_16, R_4,  R_16
};

//------------------------------------------------------------------------------
// VIC to Indexc table defines which VideoModeTable entry is appropreate for this VIC code.
// Note: This table is valid ONLY for VIC codes in 861-D formats, NOT for HDMI_VIC codes
// or 3D codes!
//------------------------------------------------------------------------------
static unsigned char VIC2Index[] = {
	0,  0,  1,  1,  2,  3,  4,  4,  5,  5,
	7,  7,  8,  8, 10, 10, 11, 12, 12, 13,
	14, 15, 15, 16, 16, 19, 19, 20, 20, 23,
	23, 24, 25, 26, 27, 28, 28, 29, 29, 30,
	31, 32, 33, 33, 34, 34, 35, 36, 37, 37,
	38, 38, 39, 39, 40, 40, 41, 41, 42, 42
};

struct modename_to_vmode {
	unsigned char *name;
	int vmode;
};

static void
printVideoMode(void)
{
	TPI_TRACE_PRINT(("HDMI: Video mode = "));

	switch (siHdmiTx.VIC) {
	case 2:
		TPI_TRACE_PRINT(("HDMI_480P60_4X3 \n"));
		break;
	case 4:
		TPI_TRACE_PRINT(("HDMI_720P60 \n"));
		break;
	case 19:
		TPI_TRACE_PRINT(("HDMI_720P50 \n"));
		break;
	case 16:
		TPI_TRACE_PRINT(("HDMI_1080P60 \n"));
		break;
	case 31:
		TPI_TRACE_PRINT(("HDMI_1080P50 \n"));
		break;
	default:
		break;
	}
}

static int
ConvertVIC_To_VM_Index(void)
{
	unsigned char index;

	// The global VideoModeDescription contains all the information we need about
	// the Video mode for use to find its entry in the Videio mode table.
	//
	// The first issue.  The "VIC" may be a 891-D VIC code, or it might be an
	// HDMI_VIC code, or it might be a 3D code.  Each require different handling
	// to get the proper video mode table index.
	//
	if (siHdmiTx.HDMIVideoFormat == VMD_HDMIFORMAT_CEA_VIC) {
		//
		// This is a regular 861-D format VIC, so we use the VIC to Index
		// table to look up the index.
		//
		index = VIC2Index[siHdmiTx.VIC];
	} else if (siHdmiTx.HDMIVideoFormat == VMD_HDMIFORMAT_HDMI_VIC) {
		//
		// HDMI_VIC conversion is simple.  We need to subtract one because the codes start
		// with one instead of zero.  These values are from HDMI 1.4 Spec Table 8-13.
		//
		if ((siHdmiTx.VIC < 1) || (siHdmiTx.VIC > 4))
			index = DEFAULT_VIDEO_MODE;
		else
			index = (HDMI_VIC_BASE - 1) + siHdmiTx.VIC;
	} else {
		// This should never happen!  If so, default to first table entry
		index = DEFAULT_VIDEO_MODE;
	}

	return index;
}

static void
SetFormat(unsigned char *Data)
{
	ReadModifyWriteTPI(TPI_SYSTEM_CONTROL_DATA_REG,
		OUTPUT_MODE_MASK, OUTPUT_MODE_HDMI); // Set HDMI mode to allow color space conversion

	WriteBlockTPI(TPI_INPUT_FORMAT_REG, 2, Data);   // Program TPI AVI Input and Output Format
	WriteByteTPI(TPI_END_RIGHT_BAR_MSB, 0x00);  // Set last unsigned char of TPI AVI InfoFrame for TPI AVI I/O Format to take effect
}

static int
InitVideo(unsigned char TclkSel)
{
	unsigned char ModeTblIndex;
	unsigned char data[8];
	unsigned char Pattern;

	TPI_TRACE_PRINT(("HDMI: InitVideo()\n"));
	printVideoMode();
	TPI_TRACE_PRINT((" HF:%d ", (int) siHdmiTx.HDMIVideoFormat));
	TPI_TRACE_PRINT((" VIC:%d ", (int) siHdmiTx.VIC));
	TPI_TRACE_PRINT((" AR:%d ", (int) siHdmiTx.AspectRatio));
	TPI_TRACE_PRINT((" CS:%d ", (int) siHdmiTx.ColorSpace));
	TPI_TRACE_PRINT((" CD:%d ", (int) siHdmiTx.ColorDepth));
	TPI_TRACE_PRINT((" CR:%d ", (int) siHdmiTx.Colorimetry));
	TPI_TRACE_PRINT((" SM:%d ", (int) siHdmiTx.SyncMode));
	TPI_TRACE_PRINT((" TCLK:%d ", (int) siHdmiTx.TclkSel));
	TPI_TRACE_PRINT((" 3D:%d ", (int) siHdmiTx.ThreeDStructure));
	TPI_TRACE_PRINT((" 3Dx:%d\n", (int) siHdmiTx.ThreeDExtData));

	ModeTblIndex = (unsigned char)ConvertVIC_To_VM_Index();

	Pattern = (TclkSel << 6) & TWO_MSBITS;		// Use TPI 0x08[7:6] for 9022A/24A video clock multiplier
	ReadSetWriteTPI(TPI_PIX_REPETITION, Pattern);	//TClkSel1:Ratio of output TMDS clock to input video clock,00-x0.5,01- x1 (default),10 -x2,11-x4

	// Take values from VModesTable[]:
	if( (siHdmiTx.VIC == 6) || (siHdmiTx.VIC == 7) ||	//480i
	     (siHdmiTx.VIC == 21) || (siHdmiTx.VIC == 22) ) {	//576i
		data[0] = (VModesTable[ModeTblIndex].PixClk /2) & 0x00FF;
		data[1] = ((VModesTable[ModeTblIndex].PixClk /2) >> 8) & 0xFF;
	} else {
		data[0] = VModesTable[ModeTblIndex].PixClk & 0x00FF;	// write Pixel clock to TPI registers 0x00, 0x01
		data[1] = (VModesTable[ModeTblIndex].PixClk >> 8) & 0xFF;
	}
	
	data[2] = VModesTable[ModeTblIndex].Tag.VFreq & 0x00FF;		// write Vertical Frequency to TPI registers 0x02, 0x03
	data[3] = (VModesTable[ModeTblIndex].Tag.VFreq >> 8) & 0xFF;

	if( (siHdmiTx.VIC == 6) || (siHdmiTx.VIC == 7) ||	//480i
	     (siHdmiTx.VIC == 21) || (siHdmiTx.VIC == 22) ) {	//576i
		data[4] = (VModesTable[ModeTblIndex].Tag.Total.Pixels /2) & 0x00FF;	// write total number of pixels to TPI registers 0x04, 0x05
		data[5] = ((VModesTable[ModeTblIndex].Tag.Total.Pixels /2) >> 8) & 0xFF;
	} else {
		data[4] = VModesTable[ModeTblIndex].Tag.Total.Pixels & 0x00FF;	// write total number of pixels to TPI registers 0x04, 0x05
		data[5] = (VModesTable[ModeTblIndex].Tag.Total.Pixels >> 8) & 0xFF;
	}
	
	data[6] = VModesTable[ModeTblIndex].Tag.Total.Lines & 0x00FF;	// write total number of lines to TPI registers 0x06, 0x07
	data[7] = (VModesTable[ModeTblIndex].Tag.Total.Lines >> 8) & 0xFF;

	WriteBlockTPI(TPI_PIX_CLK_LSB, 8, data);	// Write TPI Mode data.//0x00-0x07 :Video Mode Defines the incoming resolution

	// TPI Input Bus and Pixel Repetition Data
	// data[0] = Reg0x08;
	data[0] = 0;  // Set to default 0 for use again
	//data[0] = (VModesTable[ModeTblIndex].PixRep) & LOW_BYTE;	// Set pixel replication field of 0x08
	data[0] |= BIT_BUS_24;						// Set 24 bit bus:Input Bus Select. The input data bus can be either one pixel wide or 1/2  pixel wide. The bit defaults to 1 to select full pixel mode. In  1/2  pixel mode, the full pixel is brought in on two successive clock edges (one rising, one falling). 
	//All parts support 24-bit full-pixel and 12-bit half-pixel input modes.
	data[0] |= (TclkSel << 6) & TWO_MSBITS;

#ifdef CLOCK_EDGE_FALLING
	data[0] &= ~BIT_EDGE_RISE;
#endif
#ifdef CLOCK_EDGE_RISING
	data[0] |= BIT_EDGE_RISE;
#endif
	siHdmiTx.tpivmode[0] = data[0]; // saved TPI Reg0x08 value.
	WriteByteTPI(TPI_PIX_REPETITION, data[0]);

	// TPI AVI Input and Output Format Data
	// data[0] = Reg0x09;
	// data[1] = Reg0x0A;
	data[0] = (((BITS_IN_RGB | BITS_IN_AUTO_RANGE) & ~BIT_EN_DITHER_10_8) &
			~BIT_EXTENDED_MODE);
	data[1] = (BITS_OUT_RGB | BITS_OUT_AUTO_RANGE);

	if ((siHdmiTx.VIC == 6) || (siHdmiTx.VIC == 7) ||    // 480i
	     (siHdmiTx.VIC == 21) || (siHdmiTx.VIC == 22) || // 576i
	     (siHdmiTx.VIC == 2) || (siHdmiTx.VIC == 3) ||   // 480p
	     (siHdmiTx.VIC == 17) ||(siHdmiTx.VIC == 18))    // 576p
		data[1] &= ~BIT_BT_709;
	else
		data[1] |= BIT_BT_709;

	siHdmiTx.tpivmode[1] = data[0]; // saved TPI Reg0x09 value
	siHdmiTx.tpivmode[2] = data[1]; // saved TPI Reg0x0A value
	SetFormat(data);

	// Number HSync pulses from VSync active edge to Video Data Period should be 20 (VS_TO_VIDEO)
	ReadClearWriteTPI(TPI_SYNC_GEN_CTRL, BIT_2);

	WriteByteTPI(TPI_YC_Input_Mode, 0x00);

	return TRUE;
}

static int
SetAVI_InfoFrames(void)
{
	unsigned char data[SIZE_AVI_INFOFRAME];
	unsigned char i;
	unsigned char tmp;
	unsigned char VModeTblIndex;

	TPI_TRACE_PRINT(("HDMI: SetAVI_InfoFrames()\n"));

	for (i = 0; i < SIZE_AVI_INFOFRAME; i++)
		data[i] = 0;

	tmp = 0;

	data[1] = (tmp << 5) & BITS_OUT_FORMAT; // AVI Byte1: Y1Y0 (output format)
	data[1] |= 0x11;  // A0 = 1; Active format identification data is present in the AVI InfoFrame. // S1:S0 = 01; Overscanned (television).

	if (siHdmiTx.Colorimetry == COLORIMETRY_709)		// BT.709
		data[2] = 0x80;		// AVI Byte2: C1C0
	else if (siHdmiTx.Colorimetry == COLORIMETRY_601)	// BT.601
		data[2] = 0x40;		// AVI Byte2: C1C0
	else {				// Carries no data
		data[2] &= ~BITS_7_6;	// colorimetry = 0
		data[3] &= ~BITS_6_5_4;	// Extended colorimetry = 0
	}

	VModeTblIndex = ConvertVIC_To_VM_Index();

	data[4] = siHdmiTx.VIC;

	//  Set the Aspect Ration info into the Infoframe Byte 2
	if (siHdmiTx.AspectRatio == VMD_ASPECT_RATIO_16x9) {
		data[2] |= _16_To_9;                          // AVI Byte2: M1M0
		// If the Video Mode table says this mode can be 4x3 OR 16x9, and we are pointing to the
		// table entry that is 4x3, then we bump to the next Video Table entry which will be for 16x9.
		if ((VModesTable[VModeTblIndex].AspectRatio == R_4or16) &&
		    (AspectRatioTable[siHdmiTx.VIC - 1] == R_4)) {
			siHdmiTx.VIC++;
			data[4]++;
		}
	} else {
		data[2] |= _4_To_3; // AVI Byte4: VIC
	}

	data[2] |= SAME_AS_AR; // AVI Byte2: R3..R1 - Set to "Same as Picture Aspect Ratio"
	data[5] = VModesTable[VModeTblIndex].PixRep; // AVI Byte5: Pixel Replication - PR3..PR0

	// Calculate AVI InfoFrame ChecKsum
	data[0] = 0x82 + 0x02 +0x0D;
	for (i = 1; i < SIZE_AVI_INFOFRAME; i++)
		data[0] += data[i];

	data[0] = 0x100 - data[0];

	// Write the Inforframe data to the TPI Infoframe registers
	WriteBlockTPI(TPI_AVI_BYTE_0, SIZE_AVI_INFOFRAME, data);

	return TRUE;
}


//------------------------------------------------------------------------------
// Function Name: SetAudioInfoFrames()
// Function Description: Load Audio InfoFrame data into registers and send to sink
//
// Accepts: (1) Channel count
//              (2) speaker configuration per CEA-861D Tables 19, 20
//              (3) Coding type: 0x09 for DSD Audio. 0 (refer to stream header) for all the rest
//              (4) Sample Frequency. Non zero for HBR only
//              (5) Audio Sample Length. Non zero for HBR only.
// Returns: TRUE
// Globals: none
//------------------------------------------------------------------------------
static int
SetAudioInfoFrames(unsigned char ChannelCount, unsigned char CodingType,
		   unsigned char SS, unsigned char Fs,
		   unsigned char SpeakerConfig)
{
	unsigned char data[SIZE_AUDIO_INFOFRAME];
	unsigned char i;

	TPI_TRACE_PRINT(("HDMI: SetAudioInfoFrames()\n"));

	for (i = 0; i < SIZE_AUDIO_INFOFRAME; i++)
		data[i] = 0;

	WriteByteTPI(MISC_INFO_FRAMES_CTRL, DISABLE_AUDIO); //  Disbale MPEG/Vendor Specific InfoFrames

	data[0] = TYPE_AUDIO_INFOFRAMES;
	data[1] = AUDIO_INFOFRAMES_VERSION;
	data[2] = AUDIO_INFOFRAMES_LENGTH;
	// Calculate checksum - 0x84 + 0x01 + 0x0A
	data[3] = TYPE_AUDIO_INFOFRAMES + AUDIO_INFOFRAMES_VERSION +
			AUDIO_INFOFRAMES_LENGTH;

	data[4] = ChannelCount;       // 0 for "Refer to Stream Header" or for 2 Channels. 0x07 for 8 Channels
	// A.M. ??? Note: this comment doesn't make sense,
	//      since ChannelCount is passed with ACHANNEL_2CH == 1,
	//	while the comment states 0 for two channels.
	//	Test later on with ChannelCount = REFER_TO_STREAM_HDR == 0 
	// 

	
	data[4] |= (CodingType << 4); // 0xC7[7:4] == 0b1001 for DSD Audio

	data[5] = ((Fs & THREE_LSBITS) << 2) | (SS & TWO_LSBITS);

	data[7] = SpeakerConfig;

	for (i = 4; i < SIZE_AUDIO_INFOFRAME; i++)
		data[3] += data[i];

	data[3] = 0x100 - data[3];

	WriteByteTPI(MISC_INFO_FRAMES_CTRL, EN_AND_RPT_AUDIO); // Re-enable Audio InfoFrame transmission and repeat

	WriteBlockTPI(MISC_INFO_FRAMES_TYPE, SIZE_AUDIO_INFOFRAME, data);

	return TRUE;
}

static void
siHdmiTx_AudioSet(void)
{
	dbg_prt();

	/* Note: Do Not(!) change the order of register writing.
	 *	 Some registers availability depends on other register
	 *	 settings.
	 */

	/* (1) - Ensure I2S stream is fed into transmitter - done outside */
	
	/* (2) Write reg 0x26 - 
	 * Audio input interface Type (temporary Disabled),
	 * HDMI output Channel Layout(2 channels),
	 * HDMI output Mute (muted),
	 * HDMI output audio-coding Type (PCM).
	 */
	WriteByteTPI(TPI_AUDIO_INTERFACE_REG, siHdmiTx.AudioMode);

	/* (3) Write reg 0x20 - TDM format, clocks polarities, MCLK multiplier */
	WriteByteTPI(TPI_I2S_IN_CFG, siHdmiTx.AudioI2SFormat);

	/* (4) Write reg 0x1F -
	 * FIFO enabling (FIFO#0-Enabled;
	 * 	All Other FIFOs - Don't touch ! - see note below).
	 * Mapped FIFO (FIFO#0),
	 * Auto down-sampling (Disabled),
	 * L/R input Swap (No swap),
	 * TDM data port selection (depend on board's wiring!) (SD0),
	 */
	WriteByteTPI(TPI_I2S_EN, (AFIFO_ENABLE | AMAPPED_FIFO_0     |
	 			  AAUTO_DNSAMPLE_OFF | ALR_SWAP_OFF |
	 			  A_MAP_PORT_SD0));
	/* Note(!): If other FIFOs get Explicitly Disabled, then there
	 *	    is No sound! A chip bug, apparently.
	 */
	//WriteByteTPI(TPI_I2S_EN, (AFIFO_DISABLE | AMAPPED_FIFO_1));
	//WriteByteTPI(TPI_I2S_EN, (AFIFO_DISABLE | AMAPPED_FIFO_2));
	//WriteByteTPI(TPI_I2S_EN, (AFIFO_DISABLE | AMAPPED_FIFO_3));

	/* (5) Write reg 0x27 -
	 * Audio input sample witdh (16 bit),
	 * Audio input sample-rate (variable),
	 * High Bit-Rate (HBR) support (Disabled)
	 */
	WriteByteTPI(TPI_AUDIO_SAMPLE_CTRL, (AINPUT_WL_16BITS	|
	 				     siHdmiTx.AudioInFs |
	 				     AHBR_DISABLED));

	/* (6) Write regs 0x21 through 0x25: Stream Header Settings */
	WriteByteTPI(TPI_I2S_CHST_0, 0x00);
	WriteByteTPI(TPI_I2S_CHST_1, 0x00);
	WriteByteTPI(TPI_I2S_CHST_2, 0x00);
	WriteByteTPI(TPI_I2S_CHST_3, siHdmiTx.AudioOutFs);
	WriteByteTPI(TPI_I2S_CHST_4, ((siHdmiTx.AudioInFs >> 3) << 4) |
				       siHdmiTx.AudioOutWrdLen);

	/* A.M. >>>>>>>> Test if this is Still Needed for SiI9022A (note: "A" chip !) */
	//
	// Oscar 20100929 added for 16bit audio noise issue
	//WriteIndexedRegister(INDEXED_PAGE_1, AUDIO_INPUT_LENGTH, siHdmiTx.AudioWordLength);
	//
	WriteIndexedRegister(INDEXED_PAGE_1, AUDIO_INPUT_LENGTH, siHdmiTx.AudioOutWrdLen);

	/* (7) Program Audio InfoFrame information:
	 * write undocumented(!) regs 0xBF through 0xCD
	 */
	SetAudioInfoFrames(REFER_TO_STREAM_HDR, REFER_TO_STREAM_HDR,
			   REFER_TO_STREAM_HDR, REFER_TO_STREAM_HDR, 0x00);

	/* (8) Write reg 0x26 again - now with:
	 * - I2S Audio interface selected
	 * - UnMuted
	 */
	ReadModifyWriteTPI(TPI_AUDIO_INTERFACE_REG, AMODE_MASK, AMODE_I2S);
	ReadModifyWriteTPI(TPI_AUDIO_INTERFACE_REG, AUDIO_MUTE_MASK, AMUTE_OFF);
}

static void
siHdmiTx_Init(void)
{
	TPI_TRACE_PRINT(("HDMI: siHdmiTx_Init()\n"));

	DisableTMDS();
	mdelay(128);

	// workaround for Bug#18128
	if (siHdmiTx.ColorDepth == VMD_COLOR_DEPTH_8BIT) {
		// Yes it is, so force 16bpps first!
		siHdmiTx.ColorDepth = VMD_COLOR_DEPTH_16BIT;
		InitVideo(siHdmiTx.TclkSel);
		// Now put it back to 8bit and go do the expected InitVideo() call
		siHdmiTx.ColorDepth = VMD_COLOR_DEPTH_8BIT;
	}
	// end workaround

	InitVideo(siHdmiTx.TclkSel); // Set PLL Multiplier to x1 upon power up

	SetAVI_InfoFrames();

	EnableTMDS();
}

static int
StartTPI(void)
{
	unsigned char devID;

	TPI_TRACE_PRINT(("HDMI: StartTPI()\n"));

	WriteByteTPI(TPI_ENABLE, 0x00); // Write "0" to 72:C7 to start HW TPI mode

	mdelay(100);

	devID = ReadByteTPI(TPI_DEVICE_ID);
	TPI_TRACE_PRINT(("devID: 0x%04x\n", devID));

	/* TPI frequency multiplication ratio */
	ReadModifyWriteIndexedRegister(INDEXED_PAGE_0, 0x82, 0x1, 0x1);

	if (devID == SII902XA_DEVICE_ID)
		return TRUE;

	TPI_TRACE_PRINT(("Unsupported TX, devID = 0x%X\n", (int)devID));

	return FALSE;
}

static int
siHdmiTx_TPI_Init(void)
{
	TPI_TRACE_PRINT(("\n>>siHdmiTx_TPI_Init()\n"));
	TPI_TRACE_PRINT(("\n%s\n", TPI_FW_VERSION));

	// Chip powers up in D2 mode.
	siHdmiTx.txPowerState = TX_POWER_STATE_D2;

	// Toggle TX reset pin
	TxHW_Reset();

	// Enable HW TPI mode, check device ID
	if (StartTPI()) {
		int tmp;

		tmp = ReadByteTPI(TPI_INTERRUPT_STATUS_REG);
		if ((tmp & HOT_PLUG_STATE) == HOT_PLUG_PIN_STATE_HIGH) {
			siHdmiTx.hdmiCableConnected = TRUE;
			siHdmiTx.lcd_display_mode = LCD_DISPLAY_HDMI;
		} else {
			siHdmiTx.hdmiCableConnected = FALSE;
			siHdmiTx.lcd_display_mode = LCD_DISPLAY_LCD;
		}

		if ((tmp & RX_SENSE_STATE) == RX_SENSE_ATTACHED)
			siHdmiTx.dsRxPoweredUp = TRUE;
		else
			siHdmiTx.dsRxPoweredUp = FALSE;

		EnableInterrupts(HOT_PLUG_EVENT | RX_SENSE_EVENT);

		return 0;
	}

	return EPERM;
}

static void
OnDownstreamRxPoweredDown(void)
{
	TPI_DEBUG_PRINT (("DSRX -> Powered Down\n"));
	siHdmiTx.dsRxPoweredUp = FALSE;

	DisableTMDS();
}

static void
OnDownstreamRxPoweredUp(void)
{
	TPI_DEBUG_PRINT (("DSRX -> Powered Up\n"));
	siHdmiTx.dsRxPoweredUp = TRUE;
}

static void
OnHdmiCableDisconnected(void)
{
	TPI_DEBUG_PRINT (("HDMI Disconnected\n"));

	siHdmiTx.hdmiCableConnected = FALSE;
	siHdmiTx.lcd_display_mode = LCD_DISPLAY_LCD;

	OnDownstreamRxPoweredDown();
}

static void
OnHdmiCableConnected(void)
{
	TPI_DEBUG_PRINT (("Cable Connected\n"));

	siHdmiTx.hdmiCableConnected = TRUE;
	siHdmiTx.lcd_display_mode = LCD_DISPLAY_HDMI;

	TPI_DEBUG_PRINT (("HDMI Sink Detected\n"));
	ReadModifyWriteTPI(TPI_SYSTEM_CONTROL_DATA_REG, OUTPUT_MODE_MASK, OUTPUT_MODE_HDMI);
}

static void
siHdmiTx_TPI_Poll(void)
{
	unsigned char InterruptStatus;

	//if (siHdmiTx.txPowerState == TX_POWER_STATE_D0)
	{
		InterruptStatus = ReadByteTPI(TPI_INTERRUPT_STATUS_REG);

		if (InterruptStatus & HOT_PLUG_EVENT) {  //judge if HPD is connected
			TPI_DEBUG_PRINT (("HPD	-> "));
			ReadSetWriteTPI(TPI_INTERRUPT_ENABLE_REG, HOT_PLUG_EVENT); // Enable HPD interrupt bit

			// Repeat this loop while cable is bouncing:
			do {
				WriteByteTPI(TPI_INTERRUPT_STATUS_REG, HOT_PLUG_EVENT);   //Write 1 to interrupt bits to clear the 'pending' status.
				mdelay(T_HPD_DELAY); // Delay for metastability protection and to help filter out connection bouncing
				InterruptStatus = ReadByteTPI(TPI_INTERRUPT_STATUS_REG); // Read Interrupt status register
			} while (InterruptStatus & HOT_PLUG_EVENT);			 // loop as long as HP interrupts recur

			if (((InterruptStatus & HOT_PLUG_STATE) >> 2) != siHdmiTx.hdmiCableConnected) {
				if (siHdmiTx.hdmiCableConnected == TRUE)
					OnHdmiCableDisconnected();
				else
					OnHdmiCableConnected();

				if (siHdmiTx.hdmiCableConnected == FALSE)
					return;
			}
		}

		// Check rx power
		if (InterruptStatus & RX_SENSE_EVENT) {
			if (siHdmiTx.hdmiCableConnected == TRUE) {
				if (siHdmiTx.dsRxPoweredUp == TRUE)
					OnDownstreamRxPoweredDown();
				else
					OnDownstreamRxPoweredUp();
			}
			mdelay(100); // Delay for metastability protection and to help filter out connection bouncing
			ClearInterrupt(RX_SENSE_EVENT);
		}

		// Check if Audio Error event has occurred:
		if (InterruptStatus & AUDIO_ERROR_EVENT) {
			//TPI_DEBUG_PRINT (("TP -> Audio Error Event\n"));
			//	The hardware handles the event without need for host intervention (PR, p. 31)
			ClearInterrupt(AUDIO_ERROR_EVENT);
		}
	}
}

static void
siHdmiTx_VideoSel(unsigned char vmode)
{
	siHdmiTx.HDMIVideoFormat	= VMD_HDMIFORMAT_CEA_VIC;
	siHdmiTx.ColorSpace		= RGB;
	siHdmiTx.ColorDepth		= VMD_COLOR_DEPTH_16BIT;
	siHdmiTx.SyncMode		= EXTERNAL_HSVSDE;

	switch (vmode) {
	case HDMI_480P60_4X3:
		siHdmiTx.VIC		= 2;
		siHdmiTx.AspectRatio	= VMD_ASPECT_RATIO_4x3;
		siHdmiTx.Colorimetry	= COLORIMETRY_601;
		siHdmiTx.TclkSel	= X1;
		break;

	case HDMI_720P60:
		siHdmiTx.VIC		= 4;
		siHdmiTx.AspectRatio	= VMD_ASPECT_RATIO_16x9;
		siHdmiTx.Colorimetry	= COLORIMETRY_709;
		siHdmiTx.TclkSel	= X1;
		break;

	case HDMI_720P50:
		siHdmiTx.VIC		= 19;
		siHdmiTx.AspectRatio	= VMD_ASPECT_RATIO_16x9;
		siHdmiTx.Colorimetry	= COLORIMETRY_709;
		siHdmiTx.TclkSel	= X1;
		break;

	case HDMI_1080P24:
		siHdmiTx.VIC		= 32;
		siHdmiTx.AspectRatio	= VMD_ASPECT_RATIO_16x9;
		siHdmiTx.Colorimetry	= COLORIMETRY_709;
		siHdmiTx.TclkSel	= X1;
		break;

	case HDMI_1080P25:
		siHdmiTx.VIC		= 33;
		siHdmiTx.AspectRatio	= VMD_ASPECT_RATIO_16x9;
		siHdmiTx.Colorimetry	= COLORIMETRY_709;
		siHdmiTx.TclkSel	= X1;
		break;

	case HDMI_1080P30:
		siHdmiTx.VIC		= 34;
		siHdmiTx.AspectRatio	= VMD_ASPECT_RATIO_16x9;
		siHdmiTx.Colorimetry	= COLORIMETRY_709;
		siHdmiTx.TclkSel	= X1;
		break;

	default:
		break;
	}
}

static struct modename_to_vmode vmodes[] = {
	{ "720p50",   HDMI_720P50 },
	{ "720p50a",  HDMI_720P50 },
	{ "720p60",   HDMI_720P60 },
	{ "720p60a",  HDMI_720P60 },
	{ "1080p24",  HDMI_1080P24 },
	{ "1080p24a", HDMI_1080P24 },
	{ "1080p25",  HDMI_1080P25 },
	{ "1080p25a", HDMI_1080P25 },
	{ "1080p30",  HDMI_1080P30 },
	{ "1080p30a", HDMI_1080P30 },
};

static void
hdmi_sii_set_videomode(struct fb_videomode *mode)
{
	int i;
	char *name, *token;

	name = kmalloc(strlen(mode->name) + 1, GFP_KERNEL);
	if (!name)
		return;
	strcpy(name, mode->name);

	while ((token = strsep(&name, ",")) != NULL) {
		for (i = 0; i < ARRAY_SIZE(vmodes); i++) {
			if (!strcmp(token, vmodes[i].name)) {
				if (siHdmiTx.txPowerState == TX_POWER_STATE_D2)
					siHdmiTx_PowerStateD0();

				siHdmiTx_VideoSel(vmodes[i].vmode);
				siHdmiTx_Init();
				kfree(name);
				return;
			}
		}
	}

	DisableTMDS();
	siHdmiTx_PowerStateD2();
	kfree(name);
}

static int
fb_notifier_callback(struct notifier_block *p,
                     unsigned long event, void *data)
{
	struct fb_event *fb_event = data;
	struct fb_info *info = fb_event->info;

	if (event == FB_EVENT_MODE_CHANGE_ALL)
		hdmi_sii_set_videomode(info->mode);

	return 0;
}

static void hdmi_switch_work(struct work_struct *work)
{
	siHdmiTx_TPI_Poll();
	if (siHdmiTx.lcd_display_mode != switch_data->status) {
		switch_data->status = siHdmiTx.lcd_display_mode;
		switch_set_state(&switch_data->sdev, siHdmiTx.lcd_display_mode);
	}
	enable_irq(sii902xA->irq);
}

static irqreturn_t sii902xA_interrupt(int irq, void *dev_id)
{
	disable_irq_nosync(irq);
	schedule_work(&switch_data->work);

	return IRQ_HANDLED;
}

static ssize_t
sii902xA_sysfs_show_mode(struct device *dev, struct device_attribute *attr,
                         char *buf)
{
	if (siHdmiTx.txPowerState == TX_POWER_STATE_D0)
		return scnprintf(buf, PAGE_SIZE, "on\n");
	else if (siHdmiTx.txPowerState == TX_POWER_STATE_D2)
		return scnprintf(buf, PAGE_SIZE, "monitor\n");
	else if (siHdmiTx.txPowerState == TX_POWER_STATE_D3)
		return scnprintf(buf, PAGE_SIZE, "off\n");

	return 0;
}

static ssize_t
sii902xA_sysfs_store_mode(struct device *dev, struct device_attribute *attr,
                          const char *buf, size_t count)
{
	/* Currently the modes are switched automatically */
	return count;
}

static DEVICE_ATTR(mode, S_IWUSR | S_IRUGO, sii902xA_sysfs_show_mode, sii902xA_sysfs_store_mode);

static struct attribute *sii902xA_attrs[] = {
	&dev_attr_mode.attr,
	NULL
};

static struct attribute_group sii902xA_attr_group = {
	.attrs = sii902xA_attrs,
};

static const struct i2c_device_id hmdi_sii_id[] = {
	{ "sii902xA", 0 },
};

#ifdef CONFIG_DEBUG_FS
static ssize_t
hdmi_sii_debugfs_read(struct file *file, char __user *userbuf, size_t count,
                      loff_t *ppos)
{
	char *buf = kmalloc(4096, GFP_KERNEL);
	ssize_t size_read = 0;
	int len = 0;
	int i;

	if (buf) {
		for (i = 0; i <= 0xFF; i++)
			len += sprintf(buf + len, "0x%02x: 0x%02x\n", i,
			               i2c_smbus_read_byte_data(sii902xA, i)) + 1;

		size_read = simple_read_from_buffer(userbuf, count, ppos, buf, len);
		kfree(buf);
	}

	return size_read;
}

static int
hdmi_sii_debugfs_open(struct inode *inode, struct file *file)
{
	file->private_data = inode->i_private;

	return 0;
}

static const struct file_operations hdmi_sii_debugfs_operations = {
	.open		= hdmi_sii_debugfs_open,
	.read		= hdmi_sii_debugfs_read,
	.llseek		= default_llseek,
};
#endif

/******************************************************************************/

/* ASoC codec driver driver (under kernel 2.6.39) for the SiI9022A chip.
 *
 * Note the "A"(!). Chips that are "non-A" need special treatment, accessing
 * internal rtegisters (using XXX_IndexedRegister() functions above).
 * This code does Not implement this special treatment!
 */


static int
sii902xA_dai_set_clkdiv(struct snd_soc_dai *dai, int div_id, int div)
{
	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static int
sii902xA_dai_set_sysclk(struct snd_soc_dai *dai, int clk_id,
			unsigned int freq, int dir)
{
	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static int
sii902xA_dai_set_pll(struct snd_soc_dai *codec_dai, int pll_id, int source,
		     unsigned int freq_in, unsigned int freq_out)
{
	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static int
sii902xA_dai_set_fmt(struct snd_soc_dai *dai, unsigned int fmt)
{
	unsigned int tdm_fmt;
	unsigned int tdm_clks_polar;
	
	dbg_prt();

	/* Check Master/Slave */
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		dbg_prt("%s", "HDMI is TDM Master. Unsupported. Aborting!");
		return -EINVAL;
		break;
		
	case SND_SOC_DAIFMT_CBS_CFS:
		dbg_prt("%s", "HDMI is TDM Slave");
		break;

	case SND_SOC_DAIFMT_CBS_CFM:
	case SND_SOC_DAIFMT_CBM_CFS:
	default:
		dbg_prt("%s",
			"Unsupported Master/Slave setting. Aborting!");
		return -EINVAL;
	}

	tdm_fmt = fmt & SND_SOC_DAIFMT_FORMAT_MASK;
	if ( (tdm_fmt != SND_SOC_DAIFMT_I2S)	 &&
	     (tdm_fmt != SND_SOC_DAIFMT_RIGHT_J) &&
	     (tdm_fmt != SND_SOC_DAIFMT_LEFT_J) ) {
		dbg_prt("%s", "Unsupported TDM format. Aborting!");
		return -EINVAL;
	}

	tdm_clks_polar = fmt & SND_SOC_DAIFMT_INV_MASK;

	/* For any of the below TDM two-time-slots formats: */
	/* Reg 0x26 */
	siHdmiTx.AudioMode = (AMODE_DISABLED | ALAYOUT_2CH | AMUTE_ON |
			      ACODE_TYP_PCM);

	/* Set Format */
	switch (tdm_fmt) {
	case SND_SOC_DAIFMT_I2S:
		dbg_prt("%s", "TDM format is standard I2S");
		/* Reg 0x20: bits 2,1,0 */
		siHdmiTx.AudioI2SFormat = (SD_JUSTIFY_LEFT	|
					   SD_DIR_MSB_FIRST	|
					   WS_SD_1_BIT_SHIFT);
		break;
	
	case SND_SOC_DAIFMT_RIGHT_J:
		dbg_prt("%s", "TDM format is Right-Justified");
		/* Reg 0x20: bits 2,1,0 */
		siHdmiTx.AudioI2SFormat = (SD_JUSTIFY_RIGHT	|
					   SD_DIR_MSB_FIRST	|
					   WS_SD_NO_BIT_SHIFT);
		break;
		
	case SND_SOC_DAIFMT_LEFT_J:
		dbg_prt("%s", "TDM format is Left-Justified");
		/* Reg 0x20: bits 2,1,0 */
		siHdmiTx.AudioI2SFormat = (SD_JUSTIFY_LEFT	|
					   SD_DIR_MSB_FIRST	|
					   WS_SD_NO_BIT_SHIFT);
		break;
	}

	/* Set TDM clocks Polarity */
	
	/* Note: "Normal/Inverted" clock-polarity interpretation
	 * Depens on the selected TDM format !
	 */
	 
	if (tdm_fmt == SND_SOC_DAIFMT_I2S) {
		/* I2S format */
		switch (tdm_clks_polar) {
		case SND_SOC_DAIFMT_NB_NF:
			/* normal BCLK(sclk); normal LRC(FSync) */
			/* Reg 0x20: bit 7: '1' - RX data latched at sclk (SCK) Rising edge
			 * 	     bit 4: '0' - Sync on Falling FSync (WS)
			 */
			siHdmiTx.AudioI2SFormat |= (SCK_SAMPLE_RISING_EDGE | WS_POALRITY_LOW);
			break;

		case SND_SOC_DAIFMT_NB_IF:
			/* normal BCLK(sclk); inverted LRC(FSync) */
			/* Reg 0x20: bit 7: '1' - RX data latched at sclk (SCK) Rising edge
			*	     bit 4: '1' - Sync on Rising FSync (WS)
			*/
			siHdmiTx.AudioI2SFormat |= (SCK_SAMPLE_RISING_EDGE | WS_POLARITY_HIGH);
			break;
			
		case SND_SOC_DAIFMT_IB_NF:
			/* inverted BCLK(sclk); normal LRC(FSync) */
			/* Reg 0x20: bit 7: '0' - RX data latched at sclk (SCK) Falling edge
			*	     bit 4: '0' - Sync on Falling FSync (WS)
			*/
			siHdmiTx.AudioI2SFormat |= (SCK_SAMPLE_FALLING_EDGE | WS_POALRITY_LOW);
			break;

		case SND_SOC_DAIFMT_IB_IF:
			/* inverted BCLK(sclk); inverted LRC(FSync) */
			/* Reg 0x20: bit 7: '0' - RX data latched at sclk (SCK) Falling edge
			*	     bit 4: '1' - Sync on Rising FSync (WS)
			*/
			siHdmiTx.AudioI2SFormat |= (SCK_SAMPLE_FALLING_EDGE | WS_POLARITY_HIGH);
			break;
		}
		
	} else if ( (tdm_fmt == SND_SOC_DAIFMT_RIGHT_J) ||
		    (tdm_fmt == SND_SOC_DAIFMT_LEFT_J) ) {
		/* Right or Left Justified format */
		switch (tdm_clks_polar) {
		case SND_SOC_DAIFMT_NB_NF:
			/* normal BCLK(sclk); normal LRC(FSync) */
			/* Reg 0x20: bit 7: '1' - RX data latched at sclk (SCK) Rising edge
			 *	     bit 4: '1' - Sync on Rising FSync (WS)
			 */
			siHdmiTx.AudioI2SFormat |= (SCK_SAMPLE_RISING_EDGE | WS_POLARITY_HIGH);
			break;
		
		case SND_SOC_DAIFMT_NB_IF:
			/* normal BCLK(sclk); inverted LRC(FSync) */
			/* Reg 0x20: bit 7: '1' - RX data latched at sclk (SCK) Rising edge
			*	     bit 4: '0' - Sync on Falling FSync (WS)
			*/
			siHdmiTx.AudioI2SFormat |= (SCK_SAMPLE_RISING_EDGE | WS_POALRITY_LOW);
			break;
			
		case SND_SOC_DAIFMT_IB_NF:
			/* inverted BCLK(sclk); normal LRC(FSync) */
			/* Reg 0x20: bit 7: '0' - RX data latched at sclk (SCK) Falling edge
			*	     bit 4: '1' - Sync on Rising FSync (WS)
			*/
			siHdmiTx.AudioI2SFormat |= (SCK_SAMPLE_FALLING_EDGE | WS_POLARITY_HIGH);
			break;
		
		case SND_SOC_DAIFMT_IB_IF:
			/* inverted BCLK(sclk); inverted LRC(FSync) */
			/* Reg 0x20: bit 7: '0' - RX data latched at sclk (SCK) Falling edge
			*	     bit 4: '0' - Sync on Falling FSync (WS)
			*/
			siHdmiTx.AudioI2SFormat |= (SCK_SAMPLE_FALLING_EDGE | WS_POALRITY_LOW);
			break;
		}
	}

	/* Set CTS MCLK multiplier */
	/* Reg 0x20 bits 6:4 */
	siHdmiTx.AudioI2SFormat |= MCLK256FS; //MCLK128FS;
	

	return 0;
}

static int
sii902xA_dai_hw_params(struct snd_pcm_substream *substream,
		       struct snd_pcm_hw_params *params,
		       struct snd_soc_dai *dai)
{
	dbg_prt("%s", DIR_STR(substream->stream));

	/* Bit width */
	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
		dbg_prt("%s", "PCM sample-length (vs. ALSA) = 16 bits; LE");
		/* Reg 0x27 bits 7:6 */
		siHdmiTx.AudioInWrdLen = AINPUT_WL_16BITS;
		/* Reg 0x25 & Internal Reg: bits 4:0 */
		siHdmiTx.AudioOutWrdLen = AOUTPUT_WL_16BITS;
		break;
	default:
		dbg_prt("%s", "Unsupported PCM sample-length. Aborting");
		return -EINVAL;
	}

	/* Sample-rate */
	switch (params_rate(params)) {
	case 32000:
		dbg_prt("%s", "fs = 32KHz");
		/* Reg 0x27 bits 7:6 */
		siHdmiTx.AudioInFs = AINPUT_FS_32K;
		/* Reg 0x25 (and Internal Reg): bits 4:0 */
		siHdmiTx.AudioOutFs = AOUTPUT_FS_32K;
		break;
	case 44100:
		dbg_prt("%s", "fs = 44.1KHz");
		/* Reg 0x27 bits 7:6 */
		siHdmiTx.AudioInFs = AINPUT_FS_44K1;
		/* Reg 0x25 (and Internal Reg): bits 4:0 */
		siHdmiTx.AudioOutFs = AOUTPUT_FS_44K1;
		break;
	case 48000:
		dbg_prt("%s", "fs = 48KHz");
		/* Reg 0x27 bits 7:6 */
		siHdmiTx.AudioInFs = AINPUT_FS_48K;
		/* Reg 0x25 (and Internal Reg): bits 4:0 */
		siHdmiTx.AudioOutFs = AOUTPUT_FS_48K;
		break;
	default:
		dbg_prt("%s %u %s",
			 "Unsupported fs:", (params_rate(params)) / 1000, "KHz. Aborting!");
		return -EINVAL;
	}

	return 0;
}

static int
sii902xA_dai_prepare(struct snd_pcm_substream *substream,
                    struct snd_soc_dai *dai)
{
	dbg_prt("%s", DIR_STR(substream->stream));

	/* Write the audio settings into the chip's registers,
	 * then Activate and Unmute
	 */
	siHdmiTx_AudioSet();

	return 0;
}

static int
sii902xA_dai_hw_free(struct snd_pcm_substream *substream,
		     struct snd_soc_dai *dai)
{
	dbg_prt("%s", DIR_STR(substream->stream));

	/* Mute */
	ReadModifyWriteTPI(TPI_AUDIO_INTERFACE_REG, AMUTE_MASK, AMUTE_ON);
	
	/* Disable Audio interface */
	ReadModifyWriteTPI(TPI_AUDIO_INTERFACE_REG, AMODE_MASK, AMODE_DISABLED);

	/* Disable all FIFOs */
	WriteByteTPI(TPI_I2S_EN, AFIFO_DISABLE | AMAPPED_FIFO_0 | A_MAP_PORT_SD0);
	WriteByteTPI(TPI_I2S_EN, AFIFO_DISABLE | AMAPPED_FIFO_1 | A_MAP_PORT_SD1);
	WriteByteTPI(TPI_I2S_EN, AFIFO_DISABLE | AMAPPED_FIFO_2 | A_MAP_PORT_SD2);
	WriteByteTPI(TPI_I2S_EN, AFIFO_DISABLE | AMAPPED_FIFO_3 | A_MAP_PORT_SD3);

	return 0;
}

static int
sii902xA_dai_trigger(struct snd_pcm_substream *substream, int cmd,
		     struct snd_soc_dai *dai)
{
	dbg_prt("%s", DIR_STR(substream->stream));
	dbg_prt("%s", "Doing nothing ---->");

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			dbg_prt("%s", "Playback Start/Resume/unPause");
		} else {
			dbg_prt("%s", "Capture Start/Resume/unPause");
		}
		break;

	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_SUSPEND:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			dbg_prt("%s", "Playback Stop/Suspend/Pause");
		} else {
			dbg_prt("%s", "Capture Stop/Suspend/Pause");
		}
		break;

	default:
		dbg_prt("%s", "Unhandled trigger");
		return -EINVAL;
	}

	return 0;
}

static int
sii902xA_dai_mute(struct snd_soc_dai *dai, int mute)
{
	dbg_prt("%s %s", "set mute", (mute) ? "On" : "Off");

	/* Mute/Unmute output HDMI sound */
	ReadModifyWriteTPI(TPI_AUDIO_INTERFACE_REG, AMUTE_MASK,
			   (mute) ? AMUTE_ON : AMUTE_OFF);

	return 0;
}

static int
sii902xA_dai_startup(struct snd_pcm_substream *substream,
		     struct snd_soc_dai *dai)
{
	dbg_prt("%s", "Doing nothing ---->");

	return 0;
}

static void
sii902xA_dai_shutdown(struct snd_pcm_substream *substream,
                     struct snd_soc_dai *dai)
{
	dbg_prt("%s", "Doing nothing ---->");
}

static struct snd_soc_dai_ops sii902xA_dai_ops = {
	.set_sysclk	= sii902xA_dai_set_sysclk,
	.set_pll	= sii902xA_dai_set_pll,
	.set_clkdiv	= sii902xA_dai_set_clkdiv,
	.set_fmt	= sii902xA_dai_set_fmt,
	.digital_mute	= sii902xA_dai_mute,
	.startup	= sii902xA_dai_startup,
	.shutdown	= sii902xA_dai_shutdown,
	.hw_params	= sii902xA_dai_hw_params,
	.hw_free	= sii902xA_dai_hw_free,
	.prepare	= sii902xA_dai_prepare,
	.trigger	= sii902xA_dai_trigger,
};

static struct snd_soc_dai_driver sii902xA_hdmi_dai = {
	.name = "sii902xA-hdmi-hifi",
	.playback = {
		.stream_name	= "Playback",
		.channels_min	= 1,
		.channels_max	= 2,
		.rates		= HDMI_RATES,
		.formats	= HDMI_FORMATS,
	},

	/* No Capture ! */

	.ops = &sii902xA_dai_ops,
};

static int
sii902xA_codec_set_bias_level(struct snd_soc_codec *codec,
			      enum snd_soc_bias_level level)
{
	dbg_prt("%s", "Doing nothing ---->");
	
	switch (level) {
	case SND_SOC_BIAS_ON:
		dbg_prt("%s", "Bias ON");
		break;
	case SND_SOC_BIAS_PREPARE:
		dbg_prt("%s", "Bias PREPARE");
		break;
	case SND_SOC_BIAS_STANDBY:
		dbg_prt("%s", "Bias STANDBY");
		break;
	case SND_SOC_BIAS_OFF:
		dbg_prt("%s", "Bias OFF");
		break;
	default:
		dbg_prt("%s %u", "Unsupported Bias level:", level);
		return -EINVAL;
	}

	/* Change to new state */
	codec->dapm.bias_level = level;

	return 0;
}


/* ASoC Controls */
static const struct snd_kcontrol_new sii902xA_snd_controls[] = {
};

/* ASoC DAPM (Dynamic Audio Power Management) Widgets */
static const struct snd_soc_dapm_widget sii902xA_dapm_widgets[] = {
};

/* ASoC audio-route Map */
static const struct snd_soc_dapm_route sii902xA_audio_route[] = {
};

static int
sii902xA_codec_probe(struct snd_soc_codec *codec)
{
	struct snd_soc_dapm_context *dapm = &codec->dapm;
	int ret = 0;

	dbg_prt();

	codec->control_data = NULL;

	sii902xA_codec_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	/* Register controls */
	ret = snd_soc_add_controls(codec, sii902xA_snd_controls,
	                           ARRAY_SIZE(sii902xA_snd_controls));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add controls: %d\n", ret);
		return ret;
	}

	/* Register the DAPM Widgets */
	ret = snd_soc_dapm_new_controls(dapm, sii902xA_dapm_widgets,
	                                ARRAY_SIZE(sii902xA_dapm_widgets));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add widgets: %d\n", ret);
		return ret;
	}

	/* Set up the HDMI audio map */
	ret = snd_soc_dapm_add_routes(dapm, sii902xA_audio_route,
	                              ARRAY_SIZE(sii902xA_audio_route));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add routes: %d\n", ret);
		return ret;
	}

	return 0;
}

static int
sii902xA_codec_remove(struct snd_soc_codec *codec)
{
	dbg_prt();

	sii902xA_codec_set_bias_level(codec, SND_SOC_BIAS_OFF);

	return 0;
}

static int
sii902xA_codec_suspend(struct snd_soc_codec *codec, pm_message_t state)
{
	dbg_prt();

	sii902xA_codec_set_bias_level(codec, SND_SOC_BIAS_OFF);

	return 0;
}

static int
sii902xA_codec_resume(struct snd_soc_codec *codec)
{
	dbg_prt();

	sii902xA_codec_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_sii902xA_hdmi = {
	.probe   = sii902xA_codec_probe,
	.remove  = sii902xA_codec_remove,
	.suspend = sii902xA_codec_suspend,
	.resume  = sii902xA_codec_resume,
	.set_bias_level = sii902xA_codec_set_bias_level,

	/* No codec registers accessed directly! */
	.reg_cache_size = 0,
	.reg_word_size = 0,
	.reg_cache_default = NULL,
};

/******************************************************************************/

static int
hdmi_sii_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret;
	struct hdmi_platform_data *pdata;

	if (!i2c_check_functionality(client->adapter,
				     I2C_FUNC_SMBUS_BYTE | I2C_FUNC_I2C))
		return -ENODEV;

	sii902xA = client;

	pdata = (struct hdmi_platform_data *)client->dev.platform_data;
	if(!pdata)
		return -EINVAL;

	siHdmiTx.lcd_display_mode = LCD_DISPLAY_LCD;

	siHdmiTx_TPI_Init();

	switch_data = kzalloc(sizeof(struct hdmi_switch_data), GFP_KERNEL);
	if (!switch_data)
		return -ENOMEM;

	switch_data->sdev.name = (&pdata->switch_platform_data)->name;
	switch_data->status = siHdmiTx.lcd_display_mode;
	ret = switch_dev_register(&switch_data->sdev);
	if (ret < 0)
		goto err_switch_dev_register;
	switch_set_state(&switch_data->sdev, siHdmiTx.lcd_display_mode);

	switch_data->attr_group = &sii902xA_attr_group;
	ret = sysfs_create_group(&switch_data->sdev.dev->kobj, &sii902xA_attr_group);
	if (ret)
		goto err_switch_dev_register;

	INIT_WORK(&switch_data->work, hdmi_switch_work);

	(void) debugfs_create_file("sii902xA", S_IFREG | S_IRUGO,
	                           NULL, NULL,
	                           &hdmi_sii_debugfs_operations);

	ret = request_irq(client->irq, sii902xA_interrupt, IRQ_TYPE_LEVEL_LOW,
	                  client->name, client);
	if (ret) {
		dev_err(&client->adapter->dev, "failed to register interrupt\n");
		goto err_switch_dev_register;
	}

	/* Register ASoC codec driver */
	ret = snd_soc_register_codec(&client->dev,
				     &soc_codec_dev_sii902xA_hdmi,
				     &sii902xA_hdmi_dai, 1);
	if (ret) {
		dev_err(&client->adapter->dev, "codec registration failed.\n");
		goto exit_irq;
	}

	notifier.notifier_call = fb_notifier_callback;
	fb_register_client(&notifier);

	dev_info(&client->adapter->dev, "registered %s successfully\n", id->name);

	return ret;

exit_irq:
	free_irq(sii902xA->irq, &sii902xA);

err_switch_dev_register:
	kfree(switch_data);

	return ret;
}

static int __devexit
hdmi_sii_remove(struct i2c_client *client)
{
	sysfs_remove_group(&switch_data->sdev.dev->kobj, switch_data->attr_group);
	switch_dev_unregister(&switch_data->sdev);
	kfree(switch_data);
	free_irq(sii902xA->irq, &sii902xA);
	dev_info(&client->adapter->dev, "detached from i2c adapter successfully\n");
	return 0;
}

static struct i2c_driver hdmi_sii_i2c_driver = {
	.driver = {
		.name = DEVICE_NAME,
		.owner = THIS_MODULE,
	},
	.probe = hdmi_sii_probe,
	.remove =  __devexit_p(hdmi_sii_remove),
	.id_table = hmdi_sii_id,
};

static int __init hdmi_sii_init(void)
{
	return i2c_add_driver(&hdmi_sii_i2c_driver);
}

static void __exit hdmi_sii_exit(void)
{
	i2c_del_driver(&hdmi_sii_i2c_driver);
}

late_initcall(hdmi_sii_init);
module_exit(hdmi_sii_exit);
MODULE_VERSION("1.4");
MODULE_AUTHOR("Silicon image SZ office, Inc.");
MODULE_DESCRIPTION("sii902xA HDMI driver");
MODULE_ALIAS("platform:hdmi-sii902xA");
MODULE_LICENSE("GPL");

