/*
 * drivers/video/ili9222fb.c - a framebuffer driver for the ILI9222 display
 *
 * Copyright (C) 2008 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/fb.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/kthread.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <linux/timer.h>
#include <linux/file.h>
#include <linux/mutex.h>
#include <linux/wait.h>
#include <asm/gpio.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#include <video/ili9222fb.h>

MODULE_LICENSE("GPL")
MODULE_AUTHOR("DSPG Technologies GmbH");
MODULE_DESCRIPTION("Framebuffer driver for the ILI9222 display");

#define ILI9222FB_NAME "ili9222fb"
static char ili9222fb_name[] = ILI9222FB_NAME;

static unsigned fps = 10;
module_param(fps, uint, 0644);
MODULE_PARM_DESC(fps, "The refresh rate in frames per second.");

static unsigned bruteforce = 0;
module_param(bruteforce, bool, 0644);
MODULE_PARM_DESC(bruteforce, "Disable the pagefault logic and simply push the"
                 "complete frame to the display every time.");

static unsigned manual_sync = 0;
module_param(manual_sync, bool, 0644);
MODULE_PARM_DESC(manual_sync, "Push changes to LCD manually.");

/**
 * struct lcdfb_mapping - data for one userspace mapping of the frame buffer
 * @link:	used to add this mapping to a list
 * @vma:	pointer to the vm area struct of that mapping
 * @refcnt:	number of userspace applications using this particular mapping
 * @faults:	number of pagefaults generated in this mapping
 * @drvdata:	pointer to the data of this driver instance
 *
 */
struct ili9222fb_mapping {
	struct list_head link;
	struct vm_area_struct *vma;
	atomic_t refcnt;
	int faults;
	struct ili9222fb *ili9222fb;
};

struct ili9222fb_transfer {
	struct list_head link;
	size_t count;
	unsigned int offset;
	unsigned short addr;
};

/**
 * struct ili9222fb - data for one instance of this driver
 * @info:		pointer to the kernel fb_info struct
 * @fb:			pointer to the vmalloc'd frame buffer
 * @num_pages:		number of physical memory pages this vmalloc uses
 * @pages:		pointers to the kernel page structs
 * @page_transfers:	for each page a pointer to a related lcdctrl_transfer
 * @thread:		refresh thread task struct pointer
 * @wq:			the refresh_timer wakes up this wait queue, all threads
 * 			register to it
 * @transfers:		list of transfer's to send to the lcd controller upon
 * 			the next refresh
 * @transfers_lock:	mutex to protect the transfers list
 * @refresh_timer:	timer which starts the next refresh by waking up wq
 * @mappings:		list of ili9222fb_mappings
 * @mappings_lock:	mutex to protect the mappings list
 *
 */
struct ili9222fb {
	void __iomem *regs;

	/* fake framebuffer */
	struct fb_info *info;
	char *fb;
	int num_pages;
	struct page **pages;
	struct ili9222fb_transfer **page_transfers;

	/* refresh vars */
	struct task_struct *thread;
	wait_queue_head_t wq;
	struct list_head transfers;
	struct mutex transfers_lock;
	struct timer_list refresh_timer;

	/* per mmap mapping */
	struct list_head mappings;
	struct mutex mappings_lock;
};


/* TODO: mark these two __devinit? */
static struct fb_fix_screeninfo ili9222fb_fix = {
	.id = ILI9222FB_NAME,
	.type = FB_TYPE_PACKED_PIXELS,
	.visual = FB_VISUAL_TRUECOLOR,
	.accel = FB_ACCEL_NONE,
	.line_length = 176 * sizeof(unsigned short),
};

static struct fb_var_screeninfo ili9222fb_var = {
	.xres = 176,
	.yres = 220,
	.xres_virtual = 176,
	.yres_virtual = 220,
	.xoffset = 0,
	.yoffset = 0,
	.bits_per_pixel = 16,

	.red = {11, 5, 0},
	.green = {5, 6, 0},
	.blue = {0, 5, 0},

	.vmode = FB_VMODE_NONINTERLACED,
	.height = 44,
	.width = 33,
};

/*
 * lcd implementation
 */

const struct ili9222fb_cmd ili9222fb_init_seq[] = {
	{ ILI9222_START_OSCILLATION                 , 0x0001 },
	{ ILI9222_DELAY                             , 10     },
	{ ILI9222_DRIVER_OUTPUT_CONTROL             , 0x091B },
	{ ILI9222_LCD_AC_DRIVEING_CONTROL           , 0x0700 },
	{ ILI9222_ENTRY_MODE                        , 0x1030 },
	{ ILI9222_COMPARE_REGISTER_1                , 0x0000 },
	{ ILI9222_COMPARE_REGISTER_2                , 0x0000 },
	{ ILI9222_DISPLAY_CONTROL_1                 , 0x0037 },
	{ ILI9222_DISPLAY_CONTROL_2                 , 0x0206 },
	{ ILI9222_DISPLAY_CONTROL_3                 , 0x0000 },
	{ ILI9222_FRAME_CYCLE_CONTROL               , 0x0000 },
	{ ILI9222_EXTERNAL_DISPLAY_INTERFACE_CONTROL, 0x0000 },
	{ ILI9222_POWER_CONTROL_1                   , 0x0000 },
	{ ILI9222_DELAY                             , 10     },
	{ ILI9222_POWER_CONTROL_2                   , 0x0000 },
	{ ILI9222_POWER_CONTROL_3                   , 0x0000 },
	{ ILI9222_DELAY                             , 400    },
	{ ILI9222_POWER_CONTROL_4                   , 0x0000 },
	{ ILI9222_DELAY                             , 40     },
	{ 0x56                                      , 0x080F },
	{ ILI9222_POWER_CONTROL_1                   , 0x4240 },
	{ ILI9222_DELAY                             , 10     },
	{ ILI9222_POWER_CONTROL_2                   , 0x0000 },
	{ ILI9222_POWER_CONTROL_3                   , 0x0014 },
	{ ILI9222_DELAY                             , 400    },
	{ ILI9222_POWER_CONTROL_4                   , 0x1319 },
	{ ILI9222_DELAY                             , 40     },
	{ ILI9222_GAMMA_CONTROL_1                   , 0x0302 },
	{ ILI9222_GAMMA_CONTROL_2                   , 0x0407 },
	{ ILI9222_GAMMA_CONTROL_3                   , 0x0304 },
	{ ILI9222_GAMMA_CONTROL_4                   , 0x0203 },
	{ ILI9222_GAMMA_CONTROL_5                   , 0x0706 },
	{ ILI9222_GAMMA_CONTROL_6                   , 0x0407 },
	{ ILI9222_GAMMA_CONTROL_7                   , 0x0706 },
	{ ILI9222_GAMMA_CONTROL_8                   , 0x0000 },
	{ ILI9222_GAMMA_CONTROL_9                   , 0x0C06 },
	{ ILI9222_GAMMA_CONTROL_10                  , 0x0F00 },
	{ ILI9222_RAM_ADDRESS_SET                   , 0x0000 },
	{ ILI9222_GATE_SCAN_CONTROL                 , 0x0000 },
	{ ILI9222_VERTICAL_SCROLL_CONTROL           , 0x0000 },
	{ ILI9222_FIRST_SCREEN_DRIVE_POSITION       , 0xDB00 },
	{ ILI9222_SECOND_SCREEN_DRIVE_POSITION      , 0xDB00 },
	{ ILI9222_HORIZONTAL_RAM_ADRESS_POSITION    , 0xAF00 },
	{ ILI9222_VERTICAL_RAM_ADDRESS_POSITION     , 0xDB00 },
};

static void
ili9222fb_do_seq(struct ili9222fb *ili9222fb, const struct ili9222fb_cmd *cmd,
                 int num_cmds)
{
	int i;

	for (i = 0; i < num_cmds; i++) {
		if (cmd[i].reg == ILI9222_DELAY) {
			mdelay(cmd->val);
			continue;
		}

		writew(cmd[i].reg, ili9222fb->regs);
		writew(cmd[i].val, ili9222fb->regs + 2);
	}
}

static unsigned short
ili9222fb_offset_to_addr(unsigned offset)
{
	unsigned short addr;
	unsigned char x, y;

	/* offset to pixel offset */
	offset >>= 1;
	
	x = offset % ili9222fb_var.xres;
	y = offset / ili9222fb_var.xres;

	addr = (y << 8) | x;
	return addr;
}

static void __devinit
ili9222fb_init_lcd(struct ili9222fb *ili9222fb)
{
	int num_cmds = ARRAY_SIZE(ili9222fb_init_seq);
	ili9222fb_do_seq(ili9222fb, ili9222fb_init_seq, num_cmds);
}

/* TODO: make faster :-) */
static void
ili9222fb_do_transfer(const struct ili9222fb *ili9222fb,
                      const struct ili9222fb_transfer *transfer)
{
	int i;
	char *base;
	static unsigned foo;
	foo = !foo;

	writew(ILI9222_RAM_ADDRESS_SET, ili9222fb->regs);
	writew(transfer->addr, ili9222fb->regs + 2);

	writew(ILI9222_WRITE_DATA_TO_GRAM, ili9222fb->regs);

	base = ili9222fb->fb + transfer->offset;
	for (i = 0; i < transfer->count; i++) {
		writew(*(((unsigned short *)base) + i), ili9222fb->regs + 2);
		//writew(foo ? 0xF800 : 0x07E0, ili9222fb->regs + 2);
	}
}

static void
ili9222fb_do_transfers(const struct ili9222fb *ili9222fb,
                       const struct list_head *transfers)
{
	const struct ili9222fb_transfer *transfer;

	if (list_empty(transfers)) {
		printk(KERN_INFO "%s(): empty transfer list\n", __FUNCTION__);
	}

	list_for_each_entry(transfer, transfers, link) {
		ili9222fb_do_transfer(ili9222fb, transfer);
	}
}

/**
 * ili9222fb_blit - refreshes the whole display
 */
static void
ili9222fb_blit(struct ili9222fb *ili9222fb)
{
	int i;

	for (i = 0; i < ili9222fb->num_pages; i++)
		ili9222fb_do_transfer(ili9222fb, ili9222fb->page_transfers[i]);
}

/*
 * refresh implementation
 */

/**
 * ili9222fb_del_transfers - removes and re-inits all items in transfers
 * @transfers: list of ili9222fb_transfer's
 */
static inline void
ili9222fb_del_transfers(struct list_head *transfers)
{
	struct ili9222fb_transfer *transfer, *tmp;

	list_for_each_entry_safe(transfer, tmp, transfers, link) {
		list_del_init(&transfer->link);
	}
}

/**
 * ili9222fb_free_transfers - remotes and kfree's all items in transfers
 * @transfers: list of ili9222fb_transfer's
 */
static inline void
ili9222fb_free_transfers(struct list_head *transfers)
{
	struct ili9222fb_transfer *transfer, *tmp;

	list_for_each_entry_safe(transfer, tmp, transfers, link) {
		list_del(&transfer->link);
		kfree(transfer);
	}
}

/**
 * ili9222fb_zap_mappings - zap's the pages off the mappings
 * @mappings: list of ili9222fb_mappings
 *
 * This function iterates over all ili9222fb_mappings in mappings. If there has
 * been atleast one pagefault (faults > 0) in mapping, zap all the pages it
 * contains.
 */
static inline void
ili9222fb_zap_mappings(struct list_head *mappings)
{
	struct ili9222fb_mapping *map;

	list_for_each_entry(map, mappings, link) {
		if (!map->faults)
			continue;

		zap_page_range(map->vma, map->vma->vm_start,
			       map->vma->vm_end - map->vma->vm_start, NULL);

		map->faults = 0;
	}
}

/**
 * ili9222fb_run_timer - call timer after frame duration has elapsed
 * @timer: timer to modify
 */
static inline void
ili9222fb_run_timer(struct timer_list *timer)
{
	if (!timer_pending(timer)) {
		if (!fps)
			fps = 1;

		mod_timer(timer, jiffies + HZ / fps);
	}
}

/**
 * ili9222fb_timer - interrupt routine for the refresh_timer
 * @data: a pointer to the data of the driver instance
 *
 * Wakes up the wait_queue. This causes all refresh_thread's to continue.
 */
static void
ili9222fb_timer(unsigned long data)
{
	struct ili9222fb *ili9222fb = (struct ili9222fb *)data;
	wake_up(&ili9222fb->wq);
}

/**
 * ili9222fb_refresh_thread - pushes frame after frame to the display
 * @data: pointer to the device it should use
 *
 */
static int
ili9222fb_refresh_thread(void *data)
{
	struct ili9222fb *ili9222fb = data;

	while (!kthread_should_stop()) {
		if (!list_empty(&ili9222fb->transfers)) {
			mutex_lock(&ili9222fb->mappings_lock);
			mutex_lock(&ili9222fb->transfers_lock);

			ili9222fb_do_transfers(ili9222fb,
			                       &ili9222fb->transfers);

			if (!bruteforce) {
				ili9222fb_del_transfers(&ili9222fb->transfers);
				ili9222fb_zap_mappings(&ili9222fb->mappings);
			}

			if (bruteforce)
				ili9222fb_run_timer(&ili9222fb->refresh_timer);

			mutex_unlock(&ili9222fb->transfers_lock);
			mutex_unlock(&ili9222fb->mappings_lock);
		}

		interruptible_sleep_on(&ili9222fb->wq);
	}

	return 0;
}

/*
 * driver implementation
 */

/**
 * ili9222fb_write - the userspace interface write function
 *
 * Upon call, creates new transfers, adds them to a new list and writes them to
 * the display immedetially. New transfers and a seperate list have to be
 * created to not conflict with the concurrently running refresh_thread.
 */
static ssize_t
ili9222fb_write(struct fb_info *info, const char __user *buf, size_t count,
                loff_t *ppos)
{
	struct ili9222fb *ili9222fb;
	struct list_head transfers;
	
	unsigned off;
	int i;
	int page_start, page_end;

	/* skip if senseless :) */
	if (!count)
		return 0;

	off = *ppos;
	ili9222fb = info->par;

	if (off > info->screen_size)
		return -ENOSPC;

	if ((count + off) > info->screen_size)
		return -ENOSPC;

	/* copy from userspace to the fb memory */
	count -= copy_from_user(info->screen_base + off, buf, count);
	*ppos += count;

	/* update the display */
	INIT_LIST_HEAD(&transfers);

	page_start = off >> PAGE_SHIFT;
	page_end = (off + count) >> PAGE_SHIFT;
	
	for (i = page_start; i < page_end + 1; i++)
		list_add_tail(&ili9222fb->page_transfers[i]->link, &transfers);

	if (!list_empty(&transfers))
		ili9222fb_do_transfers(ili9222fb, &transfers);

	ili9222fb_del_transfers(&transfers);

	/* return nicely */
	if (count)
		return count;

	/* we should always be able to write atleast one byte */
	return -EFAULT;
}


static int
ili9222fb_setcolreg(unsigned regno, unsigned red, unsigned green, unsigned blue,
                    unsigned transp, struct fb_info *info)
{
	if (regno >= info->cmap.len)
		return 1;

#define CNVT_TOHW(val,width) ((((val)<<(width))+0x7FFF-(val))>>16)
	switch (info->fix.visual) {
	case FB_VISUAL_TRUECOLOR:
	case FB_VISUAL_PSEUDOCOLOR:
		red = CNVT_TOHW(red, info->var.red.length);
		green = CNVT_TOHW(green, info->var.green.length);
		blue = CNVT_TOHW(blue, info->var.blue.length);
		transp = CNVT_TOHW(transp, info->var.transp.length);
		break;

	case FB_VISUAL_DIRECTCOLOR:
		red = CNVT_TOHW(red, 8);	/* expect 8 bit DAC */
		green = CNVT_TOHW(green, 8);
		blue = CNVT_TOHW(blue, 8);
		/* hey, there is bug in transp handling... */
		transp = CNVT_TOHW(transp, 8);
		break;
	}
#undef CNVT_TOHW

	/* Truecolor has hardware independent palette */
	if (info->fix.visual == FB_VISUAL_TRUECOLOR) {
		u32 v;

		if (regno >= 16)
			return 1;

		v = (red << info->var.red.offset) |
		    (green << info->var.green.offset) |
		    (blue << info->var.blue.offset) |
		    (transp << info->var.transp.offset);

		switch (info->var.bits_per_pixel) {
		case 8:
			break;
		case 16:
			((u32 *) (info->pseudo_palette))[regno] = v;
			break;
		case 24:
		case 32:
			((u32 *) (info->pseudo_palette))[regno] = v;
			break;
		}
		return 0;
	}
	return 0;
}

/*
 * mmap implementation
 */

static void ili9222fb_vm_open(struct vm_area_struct *vma)
{
	struct ili9222fb_mapping *map = vma->vm_private_data;

	/* --------------------------------------------------------------------
	 * Increment the mapping's reference count
	 * ---- */
	atomic_inc(&map->refcnt);
}

static void ili9222fb_vm_close(struct vm_area_struct *vma)
{
	struct ili9222fb_mapping *map = vma->vm_private_data;
	struct ili9222fb *ili9222fb = map->ili9222fb;

	mutex_lock(&ili9222fb->mappings_lock);

	/* --------------------------------------------------------------------
	 * Decrement the reference count, and if 0, delete the mapping struct
	 * ---- */
	if (atomic_dec_and_test(&map->refcnt)) {
		list_del(&map->link);
		kfree(map);
	}

	mutex_unlock(&ili9222fb->mappings_lock);
}


static __inline__ int
ili9222fb_vm_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
	struct ili9222fb_mapping *map = vma->vm_private_data;
	struct ili9222fb *ili9222fb = map->ili9222fb;

	struct page *page;
	unsigned offset = (unsigned long)vmf->virtual_address - vma->vm_start;
	unsigned page_num = offset >> PAGE_SHIFT;

	if (page_num >= ili9222fb->num_pages)
		return VM_FAULT_SIGBUS;

	/* make page available and remember that there has been a fault */
	page = ili9222fb->pages[page_num];
	get_page(page);
	vmf->page = page;

	mutex_lock(&ili9222fb->mappings_lock);
	map->faults++;
	mutex_unlock(&ili9222fb->mappings_lock);

	/*
	 * Add the transfer corresponding to the requested page to the refresh
	 * list (if not already done so), and turn on the refresh timer (if not
	 * already running).
	 */
	mutex_lock(&ili9222fb->transfers_lock);
	
	if (likely(list_empty(&ili9222fb->page_transfers[page_num]->link)))
		list_add_tail(&ili9222fb->page_transfers[page_num]->link,
			      &ili9222fb->transfers);

	mutex_unlock(&ili9222fb->transfers_lock);

	if (!manual_sync)
		ili9222fb_run_timer(&ili9222fb->refresh_timer);

	return 0;
}

struct vm_operations_struct ili9222fb_vm_ops = {
	.open = ili9222fb_vm_open,
	.close = ili9222fb_vm_close,
	.fault = ili9222fb_vm_fault,
};

static int
ili9222fb_mmap(struct fb_info *info, struct vm_area_struct *vma)
{
	struct ili9222fb *ili9222fb = info->par;
	struct ili9222fb_mapping *map;
	int num_pages;

	if (!(vma->vm_flags & VM_WRITE)) {
		printk("%s(): need writable mapping\n", __FUNCTION__);
		return -EINVAL;
	}
	if (!(vma->vm_flags & VM_SHARED)) {
		printk("%s(): need shared mapping\n", __FUNCTION__);
		return -EINVAL;
	}
	if (vma->vm_pgoff != 0) {
		printk("%s(): need offset 0 (vm_pgoff=%lu)\n", __FUNCTION__,
		       vma->vm_pgoff);
		return -EINVAL;
	}

	num_pages = (vma->vm_end - vma->vm_start + PAGE_SIZE - 1) >> PAGE_SHIFT;
	if (num_pages > ili9222fb->num_pages) {
		printk("%s(): mapping to big (%ld > %lu)\n", __FUNCTION__,
		       vma->vm_end - vma->vm_start, info->screen_size);
		return -EINVAL;
	}

	if (vma->vm_ops) {
		printk("%s(): vm_ops already set\n", __FUNCTION__);
		return -EINVAL;
	}

	map = kmalloc(sizeof(*map), GFP_KERNEL);
	if (map == NULL) {
		printk("%s(): out of memory\n", __FUNCTION__);
		return -ENOMEM;
	}

	map->vma = vma;
	map->faults = 0;
	map->ili9222fb = ili9222fb;
	atomic_set(&map->refcnt, 1);

	mutex_lock(&ili9222fb->mappings_lock);
	list_add(&map->link, &ili9222fb->mappings);
	mutex_unlock(&ili9222fb->mappings_lock);

	vma->vm_ops = &ili9222fb_vm_ops;
	vma->vm_flags |= VM_RESERVED | VM_DONTEXPAND;
	vma->vm_private_data = map;

	return 0;
}

static int
ili9222fb_ioctl(struct fb_info *info, unsigned int cmd, unsigned long arg)
{
	struct ili9222fb *ili9222fb = info->par;
	int ret = 0;
	
	switch (cmd) {

	/* manual sync */
	case 0x46DD:
		wake_up(&ili9222fb->wq);
		break;
	
	default:
		ret = -EINVAL;
	}

	return ret;
}

static struct fb_ops ili9222fb_ops = {
	.owner = THIS_MODULE,
	.fb_write = ili9222fb_write,
	.fb_setcolreg = ili9222fb_setcolreg,
	.fb_fillrect = cfb_fillrect,
	.fb_imageblit = cfb_imageblit,
	.fb_copyarea = cfb_copyarea,
	.fb_mmap = ili9222fb_mmap,
	.fb_ioctl = ili9222fb_ioctl,
};

static int __devinit
ili9222fb_probe(struct platform_device *pdev)
{
	int ret;
	struct ili9222fb *ili9222fb;
	struct ili9222fb_pdata *pdata;
	struct resource *res;
	struct fb_info *info;
	int i;

	printk(KERN_INFO "Ilitek ILI9222 FB Driver\n");

	if (!pdev->dev.platform_data) {
		dev_err(&pdev->dev, "need valid platform data\n");
		return -ENODEV;
	}

	pdata = pdev->dev.platform_data;		

	if (pdev->num_resources != 1) {
		dev_err(&pdev->dev, "need one resource for registers\n");
		return -ENODEV;
	}
	
	res = pdev->resource;

	if (res->flags != IORESOURCE_MEM) {
		dev_err(&pdev->dev, "invalid resource type\n");
		return -ENODEV;
	}

	if (!request_mem_region(res->start, res->end - res->start,
	                        "ili9222fb_regs")) {
		dev_err(&pdev->dev, "request_mem_region failed\n");
		return -EBUSY;
	}

	/* allocate driver data */
	ili9222fb = kmalloc(sizeof(*ili9222fb), GFP_KERNEL);
	if (!ili9222fb)
		return -ENOMEM;

	memzero(ili9222fb, sizeof(*ili9222fb));
	INIT_LIST_HEAD(&ili9222fb->transfers);
	INIT_LIST_HEAD(&ili9222fb->mappings);

	mutex_init(&ili9222fb->transfers_lock);
	mutex_init(&ili9222fb->mappings_lock);

	init_waitqueue_head(&ili9222fb->wq);

	init_timer(&ili9222fb->refresh_timer);
	ili9222fb->refresh_timer.function = ili9222fb_timer;
	ili9222fb->refresh_timer.data = (unsigned long)ili9222fb;

	/* setup framebuffer */
	ret = -ENOMEM;
	info = framebuffer_alloc(sizeof(u32) * 256, &pdev->dev);
	if (!info)
		goto err_ili9222fb;

	info->pseudo_palette = info->par;
	info->par = ili9222fb;
	info->flags = FBINFO_FLAG_DEFAULT;

	info->var = ili9222fb_var;
	info->fix = ili9222fb_fix;

	info->screen_size = info->fix.line_length * info->var.yres_virtual;
	info->fix.smem_len = info->screen_size;
	info->fbops = &ili9222fb_ops;

	ret = fb_alloc_cmap(&info->cmap, 256, 0);
	if (ret < 0)
		goto err_info;

	/* attach info to driver data */
	ili9222fb->info = info;

	/* init the hardware */
	ili9222fb->regs = ioremap_nocache(res->start, res->end - res->start);
	if (!ili9222fb->regs) {
		dev_err(&pdev->dev, "unable to map registers\n");
		ret = -ENOMEM;
		goto out;
	}

	ret = gpio_request(pdata->backlight_gpio, ili9222fb_name);
	if (ret < 0)
		goto out;
	
	gpio_direction_output(pdata->backlight_gpio, 1);
	
	ili9222fb_init_lcd(ili9222fb);
	
	/* setup fake framebuffer */
	ret = -ENOMEM;
	ili9222fb->fb = vmalloc(info->screen_size);
	if (!ili9222fb->fb)
		goto err_info;

	memzero(ili9222fb->fb, info->screen_size);
	
	info->screen_base = ili9222fb->fb;

	/*
	 * Get the number of memory pages used for the fake fb, and store
	 * references to them so we can fasten up the fault handler
	 */
	ili9222fb->num_pages = (info->screen_size + PAGE_SIZE-1) >> PAGE_SHIFT;
	ili9222fb->pages = kmalloc(sizeof(ili9222fb->pages[0]) *
	                                  ili9222fb->num_pages, GFP_KERNEL);

	if (!ili9222fb->pages)
		goto err_fb;

	for (i = 0; i < ili9222fb->num_pages; i++)
		ili9222fb->pages[i] =
			vmalloc_to_page(ili9222fb->fb + i * PAGE_SIZE);

	/*
	 * For each page, create a transfer and store the reference to it. Will
	 * be used later in the refresh process.
	 */
	ili9222fb->page_transfers =
		kmalloc(sizeof(ili9222fb->page_transfers[0]) *
		        ili9222fb->num_pages, GFP_KERNEL);

	if (!ili9222fb->page_transfers)
		goto err_pages;

	memzero(ili9222fb->page_transfers,
	        sizeof(struct ili9222fb_transfer *) * ili9222fb->num_pages);

	for (i = 0; i < ili9222fb->num_pages; i++) {
		struct ili9222fb_transfer *t;
		t = kmalloc(sizeof(*(ili9222fb->page_transfers[0])),
		            GFP_KERNEL);

		/* count in pixels, not bytes */
		t->count = PAGE_SIZE >> 1;
		t->offset = i << PAGE_SHIFT;
		t->addr = ili9222fb_offset_to_addr(i << PAGE_SHIFT);

		ili9222fb->page_transfers[i] = t;
	}
	/* correct the pixelcount of the last transfer */
	ili9222fb->page_transfers[i-1]->count =
		info->screen_size - ((ili9222fb->num_pages - 1) << PAGE_SHIFT);
	ili9222fb->page_transfers[i-1]->count >>= 1;
	
	/* refresh display once to get rid of old/random data */
	ili9222fb_blit(ili9222fb);

	/* attach driver data to the device */
	dev_set_drvdata(&pdev->dev, ili9222fb);

	/* spawn refresh thread */
	/* TODO: handle more than one display */
	ili9222fb->thread = kthread_run(ili9222fb_refresh_thread, ili9222fb,
	                                "kili9222fbd");
	if (IS_ERR(ili9222fb->thread)) {
		ret = PTR_ERR(ili9222fb->thread);
		goto err_page_transfers;
	}

	/* register framebuffer userspace device */
	ret = register_framebuffer(info);
	if (ret < 0)
		goto err_thread;

	printk(KERN_INFO
	       "fb%d: %s frame buffer device, %dx%d, %d bpp (%d:%d:%d),"
	       " %luk, %u fps\n",
	       info->node,
	       info->fix.id,
	       info->var.xres,
	       info->var.yres,
	       info->var.bits_per_pixel,
	       info->var.red.length,
	       info->var.green.length,
	       info->var.blue.length, info->screen_size >> 10, fps);

	return 0;

err_thread:
	/* TODO */
	if (ili9222fb->thread)
		kthread_stop(ili9222fb->thread);

err_page_transfers:
	for (i = 0; i < ili9222fb->num_pages; i++)
		kfree(ili9222fb->page_transfers[i]);

	kfree(ili9222fb->page_transfers);

err_pages:
	kfree(ili9222fb->pages);

err_fb:
	vfree(ili9222fb->fb);

err_info:
	framebuffer_release(ili9222fb->info);

err_ili9222fb:
	kfree(ili9222fb);
	
out:
	return ret;
}

/* TODO: this is not complete */
static int
ili9222fb_remove(struct platform_device *pdev)
{
	struct ili9222fb *ili9222fb = dev_get_drvdata(&pdev->dev);

	kthread_stop(ili9222fb->thread);
	framebuffer_release(ili9222fb->info);
	vfree(ili9222fb->fb);
	kfree(ili9222fb);

	return 0;
}

static struct platform_driver ili9222fb_driver = {
	.probe = ili9222fb_probe,
	.remove = ili9222fb_remove,
	.driver = {
		.name = ili9222fb_name,
	},
};

/*
 * module implementation
 */

static int __init
ili9222fb_init(void)
{
	return platform_driver_register(&ili9222fb_driver);
}
module_init(ili9222fb_init);

static void __exit
ili9222fb_exit(void)
{
	return platform_driver_unregister(&ili9222fb_driver);
}
module_exit(ili9222fb_exit);

