/*
 * arch/arm/mach-dmw/css/coma-tdm.c - cordless TDM kernel interface
 *
 * Using interface to Linux kernel is implemented that
 * allows sending TDM messages
 *
 * Copyright (C) 2007 NXP Semiconductors
 * Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/kthread.h>
#include "coma.h"
#include "coma-service.h"

#define SERVICE_NAME "coma-tdm"
#define SERVICE_ID   COMA_SERVICE_TDM
#define CFIFO_SIZE   1024

#define COMA_TDM_TIMEOUT (1 * HZ) /* 1 sec. */

//#define COMA_DEBUG_TDM_TEST

struct tdm_msg_grant {
	uint32_t type;
	uint32_t id;
	uint32_t cookie;
	uint32_t channels;
	uint32_t sample_size;
	uint32_t rate;
} __attribute__ ((__packed__));

struct tdm_msg_revoke {
	uint32_t type;
	uint32_t id;
	uint32_t cookie;
} __attribute__ ((__packed__));


struct tdm_msg_ack {
	uint32_t type;
	uint32_t cookie;
} __attribute__ ((__packed__));

struct tdm_msg_nack {
	uint32_t type;
	uint32_t cookie;
	int32_t  reason;
} __attribute__ ((__packed__));

enum tdm_service_message_types {
	TDM_GRANT = 0,
	TDM_REVOKE = 1,
	TDM_ACK = 2,
	TDM_NACK=3
};

static DECLARE_COMPLETION(coma_tdm_reply);
static DEFINE_MUTEX(coma_tdm_mutex);

static int initialized = 0;
static int cfifo_size;
static struct cfifo *l2c, *c2l;
static unsigned int mem_handle;

static int coma_tdm_send_synced(void *buf, size_t size)
{
	int ret;

	mutex_lock(&coma_tdm_mutex);

	ret = coma_create_message(SERVICE_ID, SERVICE_ID, NULL, 0,
	                          (unsigned char *)buf, size);
	if (ret < 0) {
		pr_err("%s: failed sending message (type=%u)\n",
		       SERVICE_NAME, *(uint32_t *)buf);
		mutex_unlock(&coma_tdm_mutex);
		return ret;
	}
	coma_signal(SERVICE_ID);

	ret = wait_for_completion_timeout(&coma_tdm_reply, COMA_TDM_TIMEOUT);
	if (ret < 0)
		pr_err("%s: ack expected, but timed out\n", SERVICE_NAME);
	else
		ret = 0;

	mutex_unlock(&coma_tdm_mutex);
	return ret;
}

int coma_tdm_grant(unsigned int id, unsigned int rate, unsigned int channels, unsigned int sample_size)
{
	int ret;
	struct tdm_msg_grant msg;

	if (!initialized) {
		pr_err("%s: granting but not initialized\n", SERVICE_NAME);
		return -EFAULT;
	}

	msg.type 	= TDM_GRANT;
	msg.id 		= id;
	msg.rate 	= rate;
	msg.channels 	= channels;
	msg.sample_size = sample_size;
	msg.cookie	= 0; /* future use */

	pr_debug("%s: granting tdm %u, rate %d, channels %d, sample_size %d\n", SERVICE_NAME, id, rate, channels, sample_size);

	ret = coma_tdm_send_synced(&msg, sizeof(msg));
	if (ret < 0)
		pr_err("%s: granting tdm %u failed (%d)\n",
		       SERVICE_NAME, id, ret);
	return ret;
}
EXPORT_SYMBOL(coma_tdm_grant);

int coma_tdm_revoke(unsigned int id)
{
	int ret;
	struct tdm_msg_revoke msg;

	if (!initialized) {
		pr_err("%s: revoking but not initialized\n", SERVICE_NAME);
		return -EFAULT;
	}

	msg.type = TDM_REVOKE;
	msg.id = id;
	msg.cookie = 0; /* for future use */

	pr_debug("%s: revoking tdm %u\n", SERVICE_NAME, id);

	ret = coma_tdm_send_synced(&msg, sizeof(msg));
	if (ret < 0)
		pr_err("%s: revoking tdm %u failed (%d)\n",
		       SERVICE_NAME, id, ret);
	return ret;
}
EXPORT_SYMBOL(coma_tdm_revoke);

#ifdef COMA_DEBUG_TDM_TEST
static int coma_tdm_test(void *data)
{
	ssleep(5);
	while (1) {
		ssleep(10);
		printk("tdm_thread_test 1\n");
		coma_tdm_grant(2,16000,1,16);
		ssleep(10);
		printk("tdm_thread_test 2\n");
		coma_tdm_revoke(2);
	}

	return 0;
}
#endif

static int coma_tdm_process_message(struct cmsg *cmsg)
{
	struct tdm_msg_ack *ack = cmsg->payload;

	pr_debug("coma_tdm_process_message: cmsg->type %d, cmsg->payload_size %d, ack->type %d\n", cmsg->type, cmsg->payload_size, ack->type);

	BUG_ON(cmsg->type != SERVICE_ID);
	BUG_ON(cmsg->payload_size != sizeof(*ack) && cmsg->payload_size != sizeof(struct tdm_msg_nack));
	BUG_ON( ( (ack->type != TDM_ACK) && (ack->type != TDM_NACK) ) );

	/* do not continue in case of nack, this should be debugged */
	if (ack->type == TDM_NACK)
	{
		struct tdm_msg_nack *nack = cmsg->payload;
		pr_err("%s: received nack, reason id %d \n",SERVICE_NAME, nack->reason);
		BUG_ON(1);
	}
	complete(&coma_tdm_reply);
	return 0;
}

static int coma_tdm_setup(void)
{
	pr_debug("%s()\n", __func__);
	initialized = 1;
	return 0;
}

static void coma_tdm_remove(void)
{
	pr_debug("%s()\n", __func__);
	initialized = 0;
}

void tdm_service_init(void)
{
	int ret;
#ifdef COMA_DEBUG_TDM_TEST
	struct task_struct *tester;
#endif

	if (!l2c) {
		cfifo_size =
			cfifo_alloc(CFIFO_SIZE, &l2c, CFIFO_SIZE, &c2l, &mem_handle);
		if (cfifo_size < 0) {
			pr_err("%s: error: cannot alloc cfifos\n", SERVICE_NAME);
			return;
		}
	} else {
		cfifo_reset(l2c);
		cfifo_reset(c2l);
	}

	ret = coma_register(SERVICE_ID, SERVICE_NAME, l2c, c2l,
	                    coma_tdm_process_message,
	                    coma_tdm_setup,
	                    coma_tdm_remove);
	if (ret < 0) {
		pr_err("%s: error: cannot register service\n", SERVICE_NAME);
		cfifo_free((char *)l2c, cfifo_size, mem_handle);
	}

	mutex_init(&coma_tdm_mutex);

#ifdef COMA_DEBUG_TDM_TEST
	tester = kthread_create(coma_tdm_test, 0, "coma-tdm-test");
	wake_up_process(tester);
#endif
}

void tdm_service_exit(void)
{
	coma_deregister(SERVICE_ID);
//	cfifo_free((char *)l2c, cfifo_size, mem_handle);
}
