/**
 *  css.c
 *
 *  Copyright (C) 2012 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
#include <linux/clk.h>
#include <linux/debugfs.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/err.h>
#include <linux/firmware.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/stat.h>
#include <linux/sysfs.h>
#include <linux/uaccess.h>
#include <linux/vmalloc.h>
#include <linux/workqueue.h>

#include <mach/hardware.h>
#include <mach/platform.h>
#include <mach/css.h>
#include <mach/sec.h>
#include <mach/powerdomain.h>
#include <mach/reset.h>

#include "coma.h"
#include "../pm.h"

MODULE_AUTHOR("DSP Group Inc.");
MODULE_LICENSE("GPL");

static char *dev_clock_names[_CSS_CLK_LAST] = {
	"adpcm",
	"bmp",
	"gdmac",
	"timer1",
	"timer2",
	"uart?",
	"rtc",
	"clkout",
};

static int reset_map[_CSS_RST_LAST] = {
	RESET_CSS_ADPCM, /* CSS_RST_ADPCM */
	RESET_CSS_BMP, /* CSS_RST_BMP */
	RESET_CSS_GDMAC, /* CSS_RST_GDMAC */
	RESET_CSS_TIMERS, /* CSS_RST_TIMERS */
	-1, /* CSS_RST_UART */
};

struct memory {
	void __iomem *data;
	unsigned int size;
	struct resource *res;
};

struct phys_memory {
	void *data;
	unsigned int size;
	dma_addr_t phys;
};

struct loader_private {
	struct memory dtcm;
	struct memory itcm;
	struct memory ahbram;
	struct phys_memory dram;
	struct phys_memory dc;
	struct phys_memory cfifo;
#ifdef CONFIG_DEBUG_FS
	struct debugfs_blob_wrapper dram_blob;
#endif
	struct clk *clk_css;
	struct clk *clk_css_fast;
	struct clk *clk_css_slow;
	struct clk *clk_css_arm;
	struct clk *clk_css_etm;
	struct css_pdata *pdata;
	struct device *dev;
	struct dentry *pdentry;
	struct sec_msg *panic;
	struct sec_msg *pm_dram;
	struct sec_msg *pm_pdwn;
	struct sec_msg *pm_clk_cmd;
	struct sec_msg *pm_clk_arg;
	struct sec_msg *pm_rst_cmd;
	int state;
	struct clk *clk_css_devices[_CSS_CLK_LAST];
	int clk_css_dev_refcnt[_CSS_CLK_LAST];
	struct platform_device coma_device;
	struct work_struct panic_work;
	struct mutex lock;
};

#ifdef CONFIG_DEBUG_FS
struct css_mem_file {
	void *buf;
	size_t size;
};

static int css_debug_common_open(struct loader_private *p, struct file *f,
				 struct memory *mem)
{
	struct css_mem_file *mf;

	mf = kmalloc(sizeof(struct css_mem_file), GFP_KERNEL);
	if (!mf)
		return -ENOMEM;

	mf->size = mem->size;
	mf->buf = vmalloc(mem->size);
	if (!mf->buf) {
		kfree(mf);
		return -ENOMEM;
	}

	memcpy_fromio(mf->buf, mem->data, mem->size);
	f->private_data = mf;

	return 0;
}

static int css_debug_ahb_open(struct inode *inode, struct file *f)
{
	struct loader_private *p = inode->i_private;
	int ret = -EACCES;

	mutex_lock(&p->lock);

	if (p->state != CSS_FIRMWARE_LOADED && p->state != CSS_FIRMWARE_PANIC)
		goto out;

	ret = css_debug_common_open(p, f, &p->ahbram);

out:
	mutex_unlock(&p->lock);
	return ret;
}

static int css_debug_dtcm_open(struct inode *inode, struct file *f)
{
	struct loader_private *p = inode->i_private;
	int ret = -EACCES;

	mutex_lock(&p->lock);

	if (p->state != CSS_FIRMWARE_LOADED && p->state != CSS_FIRMWARE_PANIC)
		goto out;

	ret = css_debug_common_open(p, f, &p->dtcm);

out:
	mutex_unlock(&p->lock);
	return ret;
}

static int css_debug_itcm_open(struct inode *inode, struct file *f)
{
	struct loader_private *p = inode->i_private;
	int ret = -EACCES;

	mutex_lock(&p->lock);

	if (p->state != CSS_FIRMWARE_PANIC)
		goto out;

	ret = css_debug_common_open(p, f, &p->itcm);

out:
	mutex_unlock(&p->lock);
	return ret;
}

static int css_debug_cfifo_open(struct inode *inode, struct file *f)
{
	struct loader_private *p = inode->i_private;
	struct list_head *cfifo_list;
	cfifo_node_t *cfifo;
	size_t size, offset;
	struct css_mem_file *mf;
	int ret = -EACCES;

	mutex_lock(&p->lock);

	if (p->state != CSS_FIRMWARE_LOADED && p->state != CSS_FIRMWARE_PANIC)
		goto out;

	cfifo_list = get_cfifo_list_head();

	size = 0;
	list_for_each_entry(cfifo, cfifo_list, list)
		size += cfifo->size;

	mf = kmalloc(sizeof(struct css_mem_file), GFP_KERNEL);
	if (!mf) {
		ret = -ENOMEM;
		goto out;
	}

	mf->size = size;
	mf->buf = vmalloc(size);
	if (!mf->buf) {
		kfree(mf);
		ret = -ENOMEM;
		goto out;
	}

	offset = 0;
	list_for_each_entry(cfifo, cfifo_list, list) {
		memcpy(mf->buf + offset, cfifo->pbuf, cfifo->size);
		offset += cfifo->size;
	}

	f->private_data = mf;

out:
	mutex_unlock(&p->lock);
	return ret;
}

static ssize_t css_debug_read(struct file *f, char __user *u, size_t size, loff_t *off)
{
	struct css_mem_file *mf = f->private_data;

	if (*off >= mf->size)
		return 0;
	if (*off + size > mf->size)
		size = mf->size - *off;

	if (copy_to_user(u, mf->buf + *off, size))
		return -EFAULT;

	*off += size;

	return size;
}

int css_debug_common_release(struct inode *inode, struct file *f)
{
	struct css_mem_file *mf = f->private_data;

	vfree(mf->buf);
	kfree(mf);

	return 0;
}


static struct file_operations debug_ahb_fops = {
	.open		= css_debug_ahb_open,
	.read		= css_debug_read,
	.release	= css_debug_common_release,
};

static struct file_operations debug_dtcm_fops = {
	.open		= css_debug_dtcm_open,
	.read		= css_debug_read,
	.release	= css_debug_common_release,
};

static struct file_operations debug_itcm_fops = {
	.open		= css_debug_itcm_open,
	.read		= css_debug_read,
	.release	= css_debug_common_release,
};


static struct file_operations debug_cfifo_fops = {
	.open		= css_debug_cfifo_open,
	.read		= css_debug_read,
	.release	= css_debug_common_release,
};


static void css_create_debugfs_files(struct loader_private *p)
{
	struct dentry *pdentry;

	pdentry = debugfs_create_dir("css", NULL);
	p->pdentry = pdentry;

	p->dram_blob.data = p->dram.data;
	p->dram_blob.size = p->dram.size;

	debugfs_create_file("ahb-ram", S_IRUSR, pdentry, p, &debug_ahb_fops);
	debugfs_create_file("itcm", S_IRUSR, pdentry, p, &debug_itcm_fops);
	debugfs_create_file("dtcm", S_IRUSR, pdentry, p, &debug_dtcm_fops);
	debugfs_create_blob("dram", S_IRUSR, pdentry, &p->dram_blob);
	debugfs_create_file("cfifo", S_IRUSR, pdentry, p, &debug_cfifo_fops);
}

static void css_remove_debugfs_files(struct loader_private *p)
{
	if (p->pdentry)
		debugfs_remove_recursive(p->pdentry);
}
#else
#define css_create_debugfs_files(p)
#define css_remove_debugfs_files(p)
#endif

static void coma_device_release(struct device *dev)
{
}

static int css_power_up(struct loader_private *p)
{
	int ret;

	ret = dmw_pwrdom_enable(DMW_PWRDOM_CSS);
	if (ret)
		return ret;

	reset_release(RESET_CSS);
	ret = clk_enable(p->clk_css);
	if (ret)
		goto err_power_down;

	reset_release(RESET_CSS_ETM);
	if (!dmw_debug_disabled()) {
		ret = clk_enable(p->clk_css_etm);
		if (ret)
			goto err_disable_css;
	}

	ret = clk_enable(p->clk_css_arm);
	if (ret)
		goto err_disable_etm;

	/* release RF from reset */
	if (gpio_is_valid(p->pdata->rf_reset))
		gpio_set_value(p->pdata->rf_reset, 1);

	return 0;

err_disable_etm:
	if (!dmw_debug_disabled())
		clk_disable(p->clk_css_etm);
err_disable_css:
	reset_set(RESET_CSS_ETM);
	clk_disable(p->clk_css);
err_power_down:
	reset_set(RESET_CSS);
	dmw_pwrdom_disable(DMW_PWRDOM_CSS);
	return ret;
}

static void css_power_down(struct loader_private *p)
{
	int i;

	/* Shut down execution on the CSS */
	clk_disable(p->clk_css_arm);
	if (!dmw_debug_disabled())
		clk_disable(p->clk_css_etm);
	reset_set(RESET_CSS_ARM9);
	reset_set(RESET_CSS_ETM);
	clk_enable(p->clk_css_etm);
	clk_enable(p->clk_css_arm);
	mdelay(1);
	clk_disable(p->clk_css_arm);
	clk_disable(p->clk_css_etm);

	/* Disable all device clocks which were left enabled */
	for (i = 0; i < _CSS_CLK_LAST; i++) {
		while (p->clk_css_dev_refcnt[i] > 0) {
			clk_disable(p->clk_css_devices[i]);
			p->clk_css_dev_refcnt[i]--;
		}
	}

	/* Put all devices back into reset */
	reset_set(RESET_CSS_ADPCM);
	reset_set(RESET_CSS_BMP);
	reset_set(RESET_CSS_GDMAC);
	reset_set(RESET_CSS_TIMERS);
	if (reset_map[CSS_RST_UART] >= 0)
		reset_set(reset_map[CSS_RST_UART]);

	/* correctly reset BMP and ADPCMs (need clock during reset) */
	clk_enable(p->clk_css_devices[CSS_CLK_ADPCM]);
	clk_enable(p->clk_css_devices[CSS_CLK_BMP]);
	mdelay(1);
	clk_disable(p->clk_css_devices[CSS_CLK_ADPCM]);
	clk_disable(p->clk_css_devices[CSS_CLK_BMP]);

	/* make sure RF is in reset */
	if (gpio_is_valid(p->pdata->rf_reset))
		gpio_set_value(p->pdata->rf_reset, 0);

	/* Release DRAM if still requested */
	dmw_css_idle();

	/* Shutdown rest of the CSS */
	clk_disable(p->clk_css);
	reset_set(RESET_CSS);
	clk_enable(p->clk_css);
	mdelay(2);
	clk_disable(p->clk_css);

	dmw_pwrdom_disable(DMW_PWRDOM_CSS);
}

static int css_realloc_dma_mem(struct loader_private *p, struct css_part_hdr *dram_hdr)
{
	struct coma_config *dc;
	dma_addr_t dc_phys;
	dma_addr_t cfifo_phys;
	int cfifo_size;

	/* Reallocate DRAM section only if it has grown */
	if (p->dram.data && dram_hdr->rsize > p->dram.size) {
		dma_free_coherent(p->dev, p->dram.size, p->dram.data, p->dram.phys);
		memset(&p->dram, 0, sizeof(p->dram));
	}

	if (!p->dram.data) {
		p->dram.data = dma_alloc_coherent(p->dev,
						  dram_hdr->rsize,
						  &p->dram.phys,
						  GFP_KERNEL);
		if (!p->dram.data)
			return -ENOMEM;

		p->dram.size = dram_hdr->rsize;
	}

	if (!p->dc.data) {
		dc = (struct coma_config *)dma_alloc_coherent(p->dev,
							  sizeof(*dc),
							  &dc_phys,
							  GFP_KERNEL);

		if (!dc)
			return -ENOMEM;

		p->dc.data = (void *)dc;
		p->dc.size = sizeof(*dc);
		p->dc.phys = dc_phys;
	}

	if (!p->cfifo.data) {
		cfifo_size = cfifo_alloc(1024, &dc->fifo_c2l, 1024, &dc->fifo_l2c, &cfifo_phys);
		if (cfifo_size < 0)
			return -ENOMEM;

		dc->fifo_c2l_phys = (struct cfifo *)dc->fifo_c2l->self_phys;
		dc->fifo_l2c_phys = (struct cfifo *)dc->fifo_l2c->self_phys;

		p->cfifo.data = (void *)dc->fifo_c2l;
		p->cfifo.size = cfifo_size;
		p->cfifo.phys = cfifo_phys;
	} else {
		cfifo_reset(((struct coma_config *)p->dc.data)->fifo_l2c);
		cfifo_reset(((struct coma_config *)p->dc.data)->fifo_c2l);
	}

	return 0;
}

static void css_free_dma_mem(struct loader_private *p)
{
	if (p->cfifo.data) {
		cfifo_free(p->cfifo.data, p->cfifo.size, p->cfifo.phys);
		memset(&p->cfifo, 0,  sizeof(p->cfifo));
	}

	if (p->dc.data) {
		dma_free_coherent(p->dev, p->dc.size, p->dc.data, p->dc.phys);
		memset(&p->dc, 0, sizeof(p->dc));
	}

	if (p->dram.data) {
		dma_free_coherent(p->dev, p->dram.size, p->dram.data, p->dram.phys);
		memset(&p->dram, 0, sizeof(p->dram));
	}
}

static void got_firmware(const struct firmware *fw, void *context)
{
	struct loader_private *p = (struct loader_private *)context;
	struct css_img_hdr *img_hdr;
	struct css_part_hdr *itcm_hdr;
	struct css_part_hdr *ahbram_hdr;
	struct css_part_hdr *dram_hdr;
	struct css_part_hdr *elf_hdr = NULL;
	uint32_t *virt_dc;
	void __iomem *virt_elf_md5;
	void __iomem *virt_boot_config;
	unsigned long csize;
	struct boot_config bootcfg;

	if (!fw) {
		p->state = CSS_FIRMWARE_NA;
		dev_err(p->dev, "Loading CSS firmware failed\n");
		return;
	}

	mutex_lock(&p->lock);
	if (css_power_up(p)) {
		dev_err(p->dev, "cannot power up CSS\n");
		mutex_unlock(&p->lock);
		return;
	}

	/* check image header */
	img_hdr = (struct css_img_hdr *)fw->data;

	if (img_hdr->parts < 3) {
		dev_err(p->dev, "invalid number of parts: %lu\n",
		       img_hdr->parts);
		goto out_err;
	}

	if (img_hdr->magic != CSS_MAGIC) {
		dev_err(p->dev, "invalid image magic\n");
		goto out_err;
	}

	if (img_hdr->img_version != CSS_IMG_VERSION) {
		dev_err(p->dev, "image version mismatch (got: v%x, "
		                "kernel expected: v%x)\n",
		                img_hdr->img_version, CSS_IMG_VERSION);
		goto out_err;
	}
	if (img_hdr->abi_version != CSS_ABI_VERSION) {
		dev_err(p->dev, "ABI mismatch (got: v%x, expected: v%x)\n",
		                img_hdr->abi_version, CSS_ABI_VERSION);
		goto out_err;
	}
	dev_info(p->dev, "loading %s\n", img_hdr->info);

	csize = sizeof(*img_hdr);
	itcm_hdr   = (struct css_part_hdr *)(fw->data + csize);
	csize += sizeof(*itcm_hdr) + itcm_hdr->size;
	if ((csize > fw->size) || (itcm_hdr->size > p->itcm.size)) {
		dev_err(p->dev, "ITCM size too big! sz=0x%08lX, max=0x%08X\n",
		        itcm_hdr->size, p->itcm.size);
		goto out_err;
	}

	ahbram_hdr = (struct css_part_hdr *)(fw->data + csize);
	csize += sizeof(*ahbram_hdr) + ahbram_hdr->size;
	if ((csize > fw->size) || (ahbram_hdr->size > p->ahbram.size)) {
		dev_err(p->dev, "AHBRAM size too big! sz=0x%08lX, max=0x%08X\n",
		        ahbram_hdr->size, p->ahbram.size);
		goto out_err;
	}

	dram_hdr   = (struct css_part_hdr *)(fw->data + csize);
	csize += sizeof(*dram_hdr) + dram_hdr->size;
	if ((csize > fw->size) || (dram_hdr->size > dram_hdr->rsize)) {
		dev_err(p->dev, "DRAM size too big! sz=0x%08lX, max=0x%08X\n",
		        dram_hdr->size, p->dram.size);
		goto out_err;
	}

	if (img_hdr->parts == 4) {
		elf_hdr = (struct css_part_hdr *)(fw->data + csize);
		csize += sizeof(*dram_hdr) + elf_hdr->size;
		if (csize > fw->size) {
			dev_err(p->dev, "ELF size too big! sz=0x%08lX\n",
			       elf_hdr->size);
			goto out_err;
		}
		dev_info(p->dev,
		         "MD5 sum of ELF file: 0x%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X"
			 "%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X\n",
		         elf_hdr->md5[0],
		         elf_hdr->md5[1],
		         elf_hdr->md5[2],
		         elf_hdr->md5[3],
		         elf_hdr->md5[4],
		         elf_hdr->md5[5],
		         elf_hdr->md5[6],
		         elf_hdr->md5[7],
		         elf_hdr->md5[8],
		         elf_hdr->md5[9],
		         elf_hdr->md5[10],
		         elf_hdr->md5[11],
		         elf_hdr->md5[12],
		         elf_hdr->md5[13],
		         elf_hdr->md5[14],
		         elf_hdr->md5[15]);
	}

	if ((itcm_hdr->dest_addr < p->itcm.res->start) ||
	    (itcm_hdr->dest_addr > (p->itcm.res->start + p->itcm.size)) ||
	    ((itcm_hdr->dest_addr + itcm_hdr->size) > (p->itcm.res->start + p->itcm.size))) {
		dev_err(p->dev, "ITCM destination pointer out of range! ptr=0x%08lX\n",
		       itcm_hdr->dest_addr);
		goto out_err;
	}

	if ((ahbram_hdr->dest_addr < p->ahbram.res->start) ||
	    (ahbram_hdr->dest_addr > (p->ahbram.res->start + p->ahbram.size)) ||
	    ((ahbram_hdr->dest_addr + ahbram_hdr->size) > (p->ahbram.res->start + p->ahbram.size))) {
		dev_err(p->dev, "AHBRAM destination pointer out of range! ptr=0x%08lX\n",
		       ahbram_hdr->dest_addr);
		goto out_err;
	}

	if ((img_hdr->dc_addr > dram_hdr->size) ||
	    ((img_hdr->dc_addr + 4) > dram_hdr->size)) {
		dev_err(p->dev, "DC pointer out of range! ptr=0x%08lX\n",
		       img_hdr->dc_addr);
		goto out_err;
	}

	if ((img_hdr->elf_md5_addr > dram_hdr->size) ||
	    ((img_hdr->elf_md5_addr + 4) > dram_hdr->size)) {
		dev_err(p->dev, "Elf MD5 pointer out of range! ptr=0x%08lX\n",
		       img_hdr->elf_md5_addr);
		goto out_err;
	}

	if (css_realloc_dma_mem(p, dram_hdr) != 0)
		goto out_err;

	/* clear memory */
	memset_io(p->itcm.data, 0, p->itcm.size);
	memset_io(p->ahbram.data, 0, p->ahbram.size);
	memset(p->dram.data, 0, p->dram.size);

	/* copy CSS code and data */
	memcpy_toio(p->itcm.data + (itcm_hdr->dest_addr - p->itcm.res->start),
	            (void *)itcm_hdr + sizeof(*itcm_hdr),
	            itcm_hdr->size);

	memcpy_toio(p->ahbram.data + (ahbram_hdr->dest_addr - p->ahbram.res->start),
	            (void *)ahbram_hdr + sizeof(*ahbram_hdr),
	            ahbram_hdr->size);

	memcpy(p->dram.data + dram_hdr->dest_addr,
	       (void *)dram_hdr + sizeof(*dram_hdr),
	       dram_hdr->size);

	/* fill coma config pointer of CSS */
	virt_dc = (uint32_t *)(p->dram.data + img_hdr->dc_addr);
	*virt_dc = p->dc.phys;

	if (img_hdr->parts == 4) {
		/* fill elf md5 sum of CSS */
		virt_elf_md5 = p->dram.data + img_hdr->elf_md5_addr;
		memcpy(virt_elf_md5, elf_hdr->md5, 16);
	}

	/* fill the boot config structure for the CSS */
	bootcfg.dram_addr = (uint32_t)p->dram.phys;
	bootcfg.dram_size = p->dram.size;
	bootcfg.uart = p->pdata->uart;
	virt_boot_config = p->itcm.data + img_hdr->boot_config;
	memcpy_toio(virt_boot_config, (void *)&bootcfg, sizeof(bootcfg));

	/* make sure the DRAM is not switched off while the CSS is booting */
	dmw_css_busy();

	/* release ARM from reset */
	clk_disable(p->clk_css_arm);
	reset_release(RESET_CSS_ARM9);
	clk_enable(p->clk_css_arm);
	mdelay(2);

	/* register COMA platform device */
	memset(&p->coma_device, 0, sizeof(p->coma_device));
	p->coma_device.name = "coma";
	p->coma_device.id = 0;
	p->coma_device.dev.release = coma_device_release;
	p->coma_device.dev.platform_data = (void *)p->dc.data;

	if (platform_device_register(&p->coma_device) < 0)
		goto out_err;

	css_create_debugfs_files(p);

	p->state = CSS_FIRMWARE_LOADED;
	dev_info(p->dev, "successfully loaded CSS firmware\n");

	goto out;

out_err:
	dev_err(p->dev, "failed to load CSS firmware\n");
	css_power_down(p);
	p->state = CSS_FIRMWARE_NA;
out:
	release_firmware(fw);
	mutex_unlock(&p->lock);
}

static ssize_t set_load(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct loader_private *p = dev_get_drvdata(dev);
	int ret = 0;
	unsigned int value = simple_strtoul(buf, NULL, 0);

	mutex_lock(&p->lock);
	if ((value >= 1) && (p->state != CSS_FIRMWARE_PENDING)) {
		if (p->state == CSS_FIRMWARE_LOADED ||
		    p->state == CSS_FIRMWARE_PANIC) {

			css_remove_debugfs_files(p);
			css_power_down(p);
			platform_device_unregister(&p->coma_device);
			p->state = CSS_FIRMWARE_NA;
		}

		ret = request_firmware_nowait(THIS_MODULE,
		                              1,
		                              "css-loader",
		                              dev,
		                              GFP_KERNEL,
		                              (void *)p,
		                              got_firmware);

		if (ret == 0)
			p->state = CSS_FIRMWARE_PENDING;
	}
	mutex_unlock(&p->lock);

	if (ret != 0)
		return 0;

	return count;
}

static ssize_t get_state(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct loader_private *p = dev_get_drvdata(dev);
	return sprintf(buf, "%d\n", p->state);
}

static void
css_panic_work(struct work_struct *work)
{
	struct loader_private *p = container_of(work, struct loader_private,
		panic_work);

	mutex_lock(&p->lock);
	if (p->state == CSS_FIRMWARE_LOADED) {
		dev_crit(p->dev, "CSS entered panic mode\n");
		p->state = CSS_FIRMWARE_PANIC;

		/* Put ARM into reset to gain access to ITCM */
		clk_disable(p->clk_css_arm);
		reset_set(RESET_CSS_ARM9);
		clk_enable(p->clk_css_arm);
		mdelay(2);

		/* make sure RF is in reset for safety */
		if (gpio_is_valid(p->pdata->rf_reset))
			gpio_set_value(p->pdata->rf_reset, 0);

		kobject_uevent(&p->dev->kobj, KOBJ_CHANGE);
	}
	mutex_unlock(&p->lock);
}

static void css_panic_handler(int id, unsigned long data, void *context)
{
	struct loader_private *p = context;

	schedule_work(&p->panic_work);
}

static void css_pm_dram_handler(int id, unsigned long data, void *context)
{
	/*
	 * From the deepest sleep state we need approx. 250us to be cack to
	 * C-code. With some slack time the threshold is somewhere around
	 * 300us.
	 */
	if (data < 300)
		dmw_css_busy();
	else
		dmw_css_idle();
}

static void css_pm_pwdn_Handler(int id, unsigned long data, void *context)
{
	struct loader_private *p = context;

	/*
	 * From the deepest sleep state we need approx. 250us to be cack to
	 * C-code. Enabling PLL4 might also take 250us. With some slack time
	 * the threshold is somewhere around 550us.
	 */
	if (data < 550)
		clk_set_parent(p->clk_css, p->clk_css_fast);
	else
		clk_set_parent(p->clk_css, p->clk_css_slow);
}

static void css_pm_clk_handler(int id, unsigned long data, void *context)
{
	struct loader_private *p = context;
	unsigned clk_num = (data >> 8) & 0xff;
	int ret = -ENOSYS;

	if (clk_num >= _CSS_CLK_LAST)
		goto out;

	if (!p->clk_css_devices[clk_num])
		goto out;

	switch (data & 0xff) {
	case CSS_CLK_REQ_ENABLE:
		ret = clk_enable(p->clk_css_devices[clk_num]);
		p->clk_css_dev_refcnt[clk_num]++;
		break;
	case CSS_CLK_REQ_DISABLE:
		clk_disable(p->clk_css_devices[clk_num]);
		p->clk_css_dev_refcnt[clk_num]--;
		ret = 0;
		break;
	case CSS_CLK_REQ_SET_RATE:
		ret = clk_set_rate(p->clk_css_devices[clk_num],
			sec_msg_read(p->pm_clk_arg));
		break;
	case CSS_CLK_REQ_GET_RATE:
		ret = clk_get_rate(p->clk_css_devices[clk_num]);
		break;
	case CSS_CLK_REQ_ROUND_RATE:
		ret = clk_round_rate(p->clk_css_devices[clk_num],
			sec_msg_read(p->pm_clk_arg));
		break;
	}

out:
	sec_msg_trigger(p->pm_clk_cmd, ret);
}

static void css_pm_rst_handler(int id, unsigned long data, void *context)
{
	unsigned rst_num = (data >> 8) & 0xff;

	if (rst_num >= _CSS_CLK_LAST)
		return;
	if (reset_map[rst_num] < 0)
		return;

	rst_num = reset_map[rst_num];

	switch (data & 0xff) {
	case CSS_RST_REQ_ENABLE:
		reset_set(rst_num);
		break;
	case CSS_RST_REQ_DISABLE:
		reset_release(rst_num);
		break;
	}
}

static struct device_attribute css_attrs[] = {
	__ATTR(load, S_IWUSR, NULL, set_load),
	__ATTR(state, S_IRUSR, get_state, NULL),
};

static int __init css_probe(struct platform_device *pdev)
{
	int ret;
	int i;
	struct resource *ahbram_res;
	struct resource *itcm_res;
	struct resource *dtcm_res;
	struct loader_private *p;

	if (!pdev->dev.platform_data) {
		dev_err(&pdev->dev, "Missing platform data!\n");
		return -EINVAL;
	}

	ahbram_res = platform_get_resource(pdev, IORESOURCE_MEM, CSS_AHBRAM);
	if (!ahbram_res)
		return -EINVAL;

	itcm_res = platform_get_resource(pdev, IORESOURCE_MEM, CSS_ITCM);
	if (!itcm_res)
		return -EINVAL;

	dtcm_res = platform_get_resource(pdev, IORESOURCE_MEM, CSS_DTCM);
	if (!dtcm_res)
		return -EINVAL;

	if (!request_mem_region(ahbram_res->start,
	                        ahbram_res->end - ahbram_res->start + 1,
	                        pdev->name)) {
		dev_err(&pdev->dev, "failed to request CSS AHBRAM memory resource\n");
		return -EBUSY;
	}

	if (!request_mem_region(itcm_res->start,
	                        itcm_res->end - itcm_res->start + 1,
	                        pdev->name)) {
		dev_err(&pdev->dev, "failed to request CSS ITCM memory resource\n");
		ret = -EBUSY;
		goto out_release_ahbram_res;
	}

	if (!request_mem_region(dtcm_res->start,
	                        dtcm_res->end - dtcm_res->start + 1,
	                        pdev->name)) {
		dev_err(&pdev->dev, "failed to request CSS DTCM memory resource\n");
		ret = -EBUSY;
		goto out_release_itcm_res;
	}

	p = kzalloc(sizeof(*p), GFP_KERNEL);
	if (!p) {
		ret = -ENOMEM;
		goto out_release_dtcm_res;
	}

	p->ahbram.res = ahbram_res;
	p->itcm.res = itcm_res;
	p->dtcm.res = dtcm_res;

	p->ahbram.size = ahbram_res->end - ahbram_res->start + 1;
	p->itcm.size = itcm_res->end - itcm_res->start + 1;
	p->dtcm.size = dtcm_res->end - dtcm_res->start + 1;

	p->pdata = (struct css_pdata *)pdev->dev.platform_data;
	p->state = CSS_FIRMWARE_NA;
	p->dev = &pdev->dev;
	INIT_WORK(&p->panic_work, css_panic_work);
	mutex_init(&p->lock);

	platform_set_drvdata(pdev, p);

	if (gpio_is_valid(p->pdata->rf_reset)) {
		ret = gpio_request(p->pdata->rf_reset, "dect-rf-reset");
		if (ret) {
			dev_err(&pdev->dev, "failed to request RF reset GPIO\n");
			goto out_free_mem;
		}

		/* keep in reset */
		gpio_direction_output(p->pdata->rf_reset, 0);
	}

	p->ahbram.data = ioremap_nocache(ahbram_res->start, p->ahbram.size);
	if (!p->ahbram.data) {
		dev_err(&pdev->dev, "failed to map CSS AHBRAM\n");
		ret = -ENOMEM;
		goto out_free_gpio;
	}

	p->itcm.data = ioremap_nocache(itcm_res->start, p->itcm.size);
	if (!p->itcm.data) {
		dev_err(&pdev->dev, "failed to map CSS ITCM\n");
		ret = -ENOMEM;
		goto out_unmap_ahbram;
	}

	p->dtcm.data = ioremap_nocache(dtcm_res->start, p->dtcm.size);
	if (!p->dtcm.data) {
		dev_err(&pdev->dev, "failed to map CSS DTCM\n");
		ret = -ENOMEM;
		goto out_unmap_itcm;
	}

	/* adapt UART clock and reset */
	if (p->pdata->uart >= 0 && p->pdata->uart <= 2) {
		dev_clock_names[CSS_CLK_UART][4] = '0' + p->pdata->uart;
		reset_map[CSS_RST_UART] = RESET_UART1 + p->pdata->uart;
	} else {
		dev_clock_names[CSS_CLK_UART] = NULL;
		reset_map[CSS_RST_UART] = -1;
	}

	ret = -ENOENT;
	p->clk_css = clk_get(&pdev->dev, "css");
	if (IS_ERR_OR_NULL(p->clk_css))
		goto out_unmap_dtcm;
	p->clk_css_fast = clk_get(&pdev->dev, "fast");
	if (IS_ERR_OR_NULL(p->clk_css_fast))
		goto out_put_clk_css;
	p->clk_css_slow = clk_get(&pdev->dev, "slow");
	if (IS_ERR_OR_NULL(p->clk_css_slow))
		goto out_put_clk_css_fast;
	p->clk_css_arm = clk_get(&pdev->dev, "arm");
	if (IS_ERR_OR_NULL(p->clk_css_arm))
		goto out_put_clk_css_slow;
	p->clk_css_etm = clk_get(&pdev->dev, "etm");
	if (IS_ERR_OR_NULL(p->clk_css_etm))
		goto out_put_clk_css_arm;
	for (i = 0; i < _CSS_CLK_LAST; i++) {
		if (!dev_clock_names[i])
			continue;

		p->clk_css_devices[i] = clk_get(&pdev->dev, dev_clock_names[i]);
		if (IS_ERR(p->clk_css_devices[i])) {
			dev_err(&pdev->dev, "could not get clock '%s'\n",
				dev_clock_names[i]);
			ret = PTR_ERR(p->clk_css_devices[i]);
			goto out_put_clk_css_etm;
		}
	}

	for (i = 0; i < ARRAY_SIZE(css_attrs); i++) {
		ret = device_create_file(&pdev->dev, &css_attrs[i]);
		if (ret) {
			while (--i >= 0)
				device_remove_file(&pdev->dev, &css_attrs[i]);
			dev_err(&pdev->dev, "failed to create css sysfs file\n");
			goto out_put_clk_css_etm;
		}
	}

	p->panic = sec_msg_register(css_panic_handler, SEC_MSG_CSS_PANIC, 0, p);
	if (IS_ERR(p->panic)) {
		dev_err(&pdev->dev, "failed to register SEC handler\n");
		ret = PTR_ERR(p->panic);
		goto out_remove_attrs;
	}

	p->pm_dram = sec_msg_register(css_pm_dram_handler, SEC_MSG_DRAM,
				      SEC_FIQ, p);
	if (IS_ERR(p->pm_dram)) {
		dev_err(&pdev->dev, "failed to register SEC handler\n");
		ret = PTR_ERR(p->panic);
		goto out_dereg_panic;
	}

	p->pm_pdwn = sec_msg_register(css_pm_pwdn_Handler, SEC_MSG_PWR_DOWN,
				      SEC_FIQ, p);
	if (IS_ERR(p->pm_pdwn)) {
		dev_err(&pdev->dev, "failed to register SEC handler\n");
		ret = PTR_ERR(p->panic);
		goto out_dereg_pm_dram;
	}

	p->pm_clk_cmd = sec_msg_register(css_pm_clk_handler, SEC_MSG_CLK_CMD,
					 SEC_FIQ, p);
	if (IS_ERR(p->pm_clk_cmd)) {
		dev_err(&pdev->dev, "failed to register SEC handler\n");
		ret = PTR_ERR(p->panic);
		goto out_dereg_pm_pdwn;
	}

	p->pm_clk_arg = sec_msg_register(NULL, SEC_MSG_CLK_ARG, SEC_OVERWRITE, p);
	if (IS_ERR(p->pm_clk_arg)) {
		dev_err(&pdev->dev, "failed to register SEC handler\n");
		ret = PTR_ERR(p->panic);
		goto out_dereg_pm_clk_cmd;
	}

	p->pm_rst_cmd = sec_msg_register(css_pm_rst_handler, SEC_MSG_RST_CMD,
					 SEC_FIQ, p);
	if (IS_ERR(p->pm_rst_cmd)) {
		dev_err(&pdev->dev, "failed to register SEC handler\n");
		ret = PTR_ERR(p->panic);
		goto out_dereg_pm_clk_arg;
	}

	return 0;

out_dereg_pm_clk_arg:
	sec_msg_deregister(p->pm_clk_arg);
out_dereg_pm_clk_cmd:
	sec_msg_deregister(p->pm_clk_cmd);
out_dereg_pm_pdwn:
	sec_msg_deregister(p->pm_pdwn);
out_dereg_pm_dram:
	sec_msg_deregister(p->pm_dram);
out_dereg_panic:
	sec_msg_deregister(p->panic);
out_remove_attrs:
	for (i = 0; i < ARRAY_SIZE(css_attrs); i++)
		device_remove_file(&pdev->dev, &css_attrs[i]);
out_put_clk_css_etm:
	for (i = 0; i < _CSS_CLK_LAST; i++)
		if (!IS_ERR_OR_NULL(p->clk_css_devices[i]))
			clk_put(p->clk_css_devices[i]);
	clk_put(p->clk_css_etm);
out_put_clk_css_arm:
	clk_put(p->clk_css_arm);
out_put_clk_css_slow:
	clk_put(p->clk_css_slow);
out_put_clk_css_fast:
	clk_put(p->clk_css_fast);
out_put_clk_css:
	clk_put(p->clk_css);
out_unmap_dtcm:
	iounmap(p->dtcm.data);
out_unmap_itcm:
	iounmap(p->itcm.data);
out_unmap_ahbram:
	iounmap(p->ahbram.data);
out_free_gpio:
	if (gpio_is_valid(p->pdata->rf_reset))
		gpio_free(p->pdata->rf_reset);
out_free_mem:
	mutex_destroy(p->lock);
	kfree(p);
out_release_dtcm_res:
	release_mem_region(dtcm_res->start, dtcm_res->end - dtcm_res->start + 1);
out_release_itcm_res:
	release_mem_region(itcm_res->start, itcm_res->end - itcm_res->start + 1);
out_release_ahbram_res:
	release_mem_region(ahbram_res->start, ahbram_res->end - ahbram_res->start + 1);
	return ret;
}


static int __exit css_remove(struct platform_device *pdev)
{
	struct loader_private *p = platform_get_drvdata(pdev);
	int i;

	if (p->state == CSS_FIRMWARE_LOADED || p->state == CSS_FIRMWARE_PANIC) {
		css_remove_debugfs_files(p);
		css_power_down(p);
		platform_device_unregister(&p->coma_device);
	}
	css_free_dma_mem(p);
	flush_work_sync(&p->panic_work);

	sec_msg_deregister(p->pm_rst_cmd);
	sec_msg_deregister(p->pm_clk_arg);
	sec_msg_deregister(p->pm_clk_cmd);
	sec_msg_deregister(p->pm_pdwn);
	sec_msg_deregister(p->pm_dram);
	sec_msg_deregister(p->panic);

	for (i = 0; i < ARRAY_SIZE(css_attrs); i++)
		device_remove_file(&pdev->dev, &css_attrs[i]);

	for (i = 0; i < _CSS_CLK_LAST; i++)
		if (p->clk_css_devices[i])
			clk_put(p->clk_css_devices[i]);

	clk_put(p->clk_css_etm);
	clk_put(p->clk_css_arm);
	clk_put(p->clk_css_slow);
	clk_put(p->clk_css_fast);
	clk_put(p->clk_css);

	iounmap(p->dtcm.data);
	iounmap(p->itcm.data);
	iounmap(p->ahbram.data);
	release_mem_region(p->dtcm.res->start,
	                   p->dtcm.res->end - p->dtcm.res->start + 1);
	release_mem_region(p->itcm.res->start,
	                   p->itcm.res->end - p->itcm.res->start + 1);
	release_mem_region(p->ahbram.res->start,
	                   p->ahbram.res->end - p->ahbram.res->start + 1);

	if (gpio_is_valid(p->pdata->rf_reset))
		gpio_free(p->pdata->rf_reset);

	mutex_destroy(p->lock);
	kfree(p);

	return 0;
}

static struct platform_driver css_driver = {
	.driver = {
		.name = "css",
		.owner = THIS_MODULE,
	},
	.remove = __exit_p(css_remove),
};

static int __init css_init(void)
{
	return platform_driver_probe(&css_driver, css_probe);
}
module_init(css_init);

static void __exit css_exit(void)
{
	platform_driver_unregister(&css_driver);
}
module_exit(css_exit);

