/*
 * dp52.c -- DP52 CODEC driver
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/slab.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>
#include <sound/tlv.h>

#include <linux/dp52.h>
#include <sound/dp52.h>


/*** A.M. We surely support playback upto Fs=48KHz. What about 96KHz ?
#define DP52_PLAY_RATES (SNDRV_PCM_RATE_8000_44100)
***/
#define DP52_PLAY_RATES (SNDRV_PCM_RATE_8000_48000)

#define DP52_CAPT_RATES (SNDRV_PCM_RATE_8000	| \
						 SNDRV_PCM_RATE_11025	| \
						 SNDRV_PCM_RATE_16000)

/*** A.M. Do we support BE ?
#define DP52_FORMATS (SNDRV_PCM_FMTBIT_S16_LE | \
	SNDRV_PCM_FMTBIT_S16_BE  | \
	SNDRV_PCM_FMTBIT_S20_3LE | \
	SNDRV_PCM_FMTBIT_S20_3BE | \
	SNDRV_PCM_FMTBIT_S24_3LE | \
	SNDRV_PCM_FMTBIT_S24_3BE | \
	SNDRV_PCM_FMTBIT_S32_LE  | \
	SNDRV_PCM_FMTBIT_S32_BE)
***/
/* DP-52 I2S format limitation is 16 (min) to 32 (max) bits per "channel".
 * The actual bits (truncate/extend) are 16 to ADC buffer and 18 bits from DAC buffer.
 */
#define DP52_FORMATS (SNDRV_PCM_FMTBIT_S16_LE	| \
					  SNDRV_PCM_FMTBIT_S20_3LE	| \
					  SNDRV_PCM_FMTBIT_S24_LE	| \
					  SNDRV_PCM_FMTBIT_S32_LE)


struct dp52_priv {
	u16 reg_cache[16];	// A.M. Place holder for now. No cache at first implementation phase.
	struct snd_soc_codec codec;
};


typedef struct
{
	u16   dp_div;       // divder for dp
	u16   i2s_div;      // divder for i2s
	u16   i2s_width;	// number of bits in i2s channel (time-slot)
	u16   gain_factor;	// require gain factor of intrpolation factor

	uint  lq_sd_clk;	// LQ SD input clock frequency
	uint  hq_sd_clk;	// HQ SD input clock frequency
	u8    Nh;			// NH (depends on Fs)
	u8    Ns;			// NS depends on Fs)
	u8    lq_div;		// LQ_DIV; is "LQ_DIV" on excel sheet
	u8    hq_div;		// HQ_DIV; is "AFE_DIV" on excel sheet
} DP_SD_coeff;


static const DP_SD_coeff dp_sd_coeff[ ] =
{
	/* dp_div  i2s_div  i2s_w  gain      lq_sd_clk  hq_sd_clk   Nh  Ns  lq_div  hq_div */
	{  0,   0,   0,      0,		0,       0,			0,  0,   0,   0  },	// kFs_NONE

	/* LQ and HQ: */
	{  36, 144, 16, 0xffff,		4096000, 4096000,	8, 16,	20,  20  },	// kFs_8000
	{  22,  88, 19, 0xffff,		5644800, 6702545,	8, 19,	14,  14  },	// kFs_11025
	{  24,  96, 16, 0xffff,		6144000, 6144000,	8, 13,	16,  13  },	// kFs_12000
	{  18,  72, 16, 0xffff,		8192000, 6144000,	6, 16,	10,  13  },	// kFs_16000

	/* HQ Only: */
	{  22,  44, 19, 0xafff,		0,       6702545,	4, 19,	 0,  22  },	// kFs_22050
	{  24,  48, 16, 0xffff,		0,       6144000,	4, 16,	 0,  48  },	// kFs_24000
	{  24,  36, 16, 0xffff,		0,       6144000,	3, 16,	 0,  36  },	// kFs_32000
	{  22,  22, 19, 0x7fff,		0,       6702545,	2, 19,	 0,  22  },	// kFs_44100
	{  24,  24, 16, 0xffff,		0,       6144000,	2, 16,	 0,  13  }, // kFs_48000
	{  24,  12, 16, 0xffff,		0,       6144000,	1, 16,	 0,  13  } 	// kFs_96000
};


static int dp52_pcm_hw_params(struct snd_pcm_substream *substream,
							  struct snd_pcm_hw_params *params,
							  struct snd_soc_dai *dai)
{
	/* To DO */
	return 0;
}


static int dp52_mute(struct snd_soc_dai *dai, int mute)
{
	u16 mute_reg_val;
	u16 dac_en_mask;
	struct snd_soc_codec *codec = dai->codec;

	/* There is no explicit "digital mute" for output path in DP_52.
	 * Can disable/enable DACs instead
	 */

	/* For HQ SD DACs */
	mute_reg_val = snd_soc_read(codec, DP52_HQ_SDS_CFG);
	dac_en_mask = DP52_HQ_SDS_CFG_SD_DACL_EN_MASK | DP52_HQ_SDS_CFG_SD_DACR_EN_MASK;

	if (mute) {
		snd_soc_write(codec, DP52_HQ_SDS_CFG, mute_reg_val & ~dac_en_mask); // Disable
	} else {
		snd_soc_write(codec, DP52_HQ_SDS_CFG, mute_reg_val | dac_en_mask);  // Enable
	}

	/* For LQ SD CODEC0 TX */
	mute_reg_val = snd_soc_read(codec, DP52_LQ_SDCFG1);
	dac_en_mask = DP52_LQ_SDCFG1_SD0TX_MASK;

	if (mute) {
		snd_soc_write(codec, DP52_LQ_SDCFG1, mute_reg_val & ~dac_en_mask); // Disable
	} else {
		snd_soc_write(codec, DP52_LQ_SDCFG1, mute_reg_val | dac_en_mask);  // Enable
	}

	return 0;
}


static int dp52_set_dai_fmt(struct snd_soc_dai *codec_dai, unsigned int fmt)
{
	/* The definitions for thr "fmt" arg are found in linux-xxx/include/sound/soc-dai.h */

	/* As TDM slave, I2S format only, the format does not imfulence the DP_52 settings:
	 * There is no configuration for the number-of-bits-per-time-slot ("channel")
	 */

	return 0;
}


static int dp52_set_dai_clkdiv(struct snd_soc_dai *codec_dai, int div_id, int div)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	u16 ccr0_val, ccr1_val;

	ccr0_val = snd_soc_read(codec, DP52_CMU_CCR0);

	switch (div_id) {

	case DP52_HQ_SD_CLKDIV_ID:
		if (div > 0) {
			ccr0_val &= ~DP52_CCR0_HQDIV_MASK;
			ccr0_val |= ((div - 1) << 0) & DP52_CCR0_HQDIV_MASK;
			/* Enable HQ clock divider */
			ccr0_val |= DP52_CCR0_HQSDCLK_EN_MASK;
		} else {
			/* Disable HQ clock divider */
			ccr0_val &= ~(DP52_CCR0_HQSDCLK_EN_MASK);
		}
		snd_soc_write(codec, DP52_CMU_CCR0, ccr0_val);
		break;

	case DP52_LQ_SD_CLKDIV_ID:
		if (div > 0) {
			ccr1_val = snd_soc_read(codec, DP52_CMU_CCR1);
			ccr1_val &= ~(DP52_CCR1_LQDIV_MASK);
			ccr1_val |= ((div - 1) << 0) & DP52_CCR1_LQDIV_MASK;
			snd_soc_write(codec, DP52_CMU_CCR1, ccr1_val);
			/* Enable LQ clock divider */
			ccr0_val |= DP52_CCR0_LQSDCLK_EN_MASK;
		} else {
			/* Disable LQ clock divider */
			ccr0_val &= ~(DP52_CCR0_LQSDCLK_EN_MASK);

		}
		snd_soc_write(codec, DP52_CMU_CCR0, ccr0_val);
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

static u32 get_OD_from_NO( u32 NO_val )
{
	u32 OD_val;

	switch (NO_val) {
	case 1:
		OD_val = 0x0;
		break;
	case 2:
		OD_val = 0x1;
		break;
	case 4:
		OD_val = 0x3;
		break;
	default:
		printk(/*KERN_DEBUG*/ "get_OD_from_NO(): Wrong NO value %u\n", NO_val);
		OD_val = 0x0;
	}

	return OD_val;
}

static int dp52_set_dai_pll(struct snd_soc_dai *codec_dai,
                            int pll_id,
                            int source,
                            unsigned int freq_in,
                            unsigned int freq_out)
{
#define No_val  (4) // For Fvco = FPLLout / NO = ~320MHz. Set OD = '11' to get NO = 4 !
#define Nr_val  (3) // For Fref = FPLLin / (NR * 2) = ~12/4 = 3MHz

	u32 Nf_val;
	u32 ccr0_val, ccr1_val;
	u32 pllcr_val = 0;

	bool bypass_pll = false;
	struct snd_soc_codec *codec = codec_dai->codec;

	if ((freq_in == 0) || (freq_out == 0) || (freq_out == freq_in)) {
		bypass_pll = true;
	}

	if (! bypass_pll) {

		/* Set PLL parameters: */

		/*
		 *	FPLLout 	  NF							FPLLout
		 *	-------  =	-------- ===> NF = (NR * NO) *	-------
		 *	FPLLin		NR * NO 						FPLLin
		 */

		Nf_val = freq_out * Nr_val * No_val; // Fits well within 32 bits. No worry.
		Nf_val /= freq_in;
		/* A.M. Note: Amos B. has fixed Nf_val, for D-Class usage: */
		//Nf_val = 168;

		/* OD */
		pllcr_val  |= (get_OD_from_NO(No_val) << 14) & DP52_PLLCR_OD_MASK;
		/* R */
		pllcr_val  |= ((Nr_val - 2) << 9) & DP52_PLLCR_R_MASK;
		/* F */
		pllcr_val  |= ((Nf_val - 2) << 0) & DP52_PLLCR_F_MASK;
		snd_soc_write(codec, DP52_CMU_PLLCR, pllcr_val);


		/* Turn On internal PLL */
		ccr0_val = snd_soc_read(codec, DP52_CMU_CCR0);
		ccr0_val |= DP52_CCR0_PLLEN_MASK;
		snd_soc_write(codec, DP52_CMU_CCR0, ccr0_val);

		/* Wait for PLL stabilization ? */

		/* Clock HQ and LQ SDs from PLL */
		ccr1_val = snd_soc_read(codec, DP52_CMU_CCR1);
		ccr1_val |= DP52_CCR1_SDCLKSEL_MASK;
		snd_soc_write(codec, DP52_CMU_CCR1, ccr1_val);


	} else {

		/* ByPass internal PLL : */

		/* Clock HQ and LQ SDs directly from FDSP ("DP clock") */
		ccr1_val = snd_soc_read(codec, DP52_CMU_CCR1);
		ccr1_val &= ~(DP52_CCR1_SDCLKSEL_MASK);
		snd_soc_write(codec, DP52_CMU_CCR1, ccr1_val);

		/* Turn Off internal PLL */
		ccr0_val = snd_soc_read(codec, DP52_CMU_CCR0);
		ccr0_val &= ~(DP52_CCR0_PLLEN_MASK);
		snd_soc_write(codec, DP52_CMU_CCR0, ccr0_val);
	}

	return 0;
}


static struct snd_soc_dai_ops dp52_dai_ops = {
	.hw_params = dp52_pcm_hw_params,
	.digital_mute = dp52_mute,
	.set_fmt = dp52_set_dai_fmt,
	.set_clkdiv = dp52_set_dai_clkdiv,
	.set_pll = dp52_set_dai_pll,
	//.set_sysclk =

};


struct snd_soc_dai dp52_dai = {
	.name = "DP52 HSSPOUT",
	.playback = {
		.stream_name = "HiFi Playback",
		.channels_min = 1,
		.channels_max = 2,
		.rates = DP52_PLAY_RATES,
		.formats = DP52_FORMATS,
	},
	.capture = {
		.stream_name = "HiFi Capture",
		.channels_min = 1,
		.channels_max = 2,
		.rates = DP52_CAPT_RATES,
		.formats = DP52_FORMATS,
	},
	.ops = &dp52_dai_ops,
};
EXPORT_SYMBOL(dp52_dai);


static const struct snd_kcontrol_new dp52_snd_controls[] = {
	/* TODO: mixer controls */
};

static const struct snd_soc_dapm_widget dp52_dapm_widgets[] = {
	/* TODO: DAPM widgets */
};

static const struct snd_soc_dapm_route dp52_dapm_paths[] = {
	/* TODO: DAPM widget routes */
};

static unsigned int dp52_read(struct snd_soc_codec *codec, unsigned int reg)
{
	int dp_reg_val;

	//u16 *cache = codec->reg_cache;

	dp_reg_val = dp52_direct_read(reg);
	return (unsigned int)dp_reg_val;
}

static int dp52_write(struct snd_soc_codec *codec, unsigned int reg, unsigned int val)
{
	//u16 *cache = codec->reg_cache;

	dp52_direct_write(reg, val);
	return 0;
}

static int dp52_set_bias_level(struct snd_soc_codec *codec,
				 enum snd_soc_bias_level level)
{
	switch (level) {
	case SND_SOC_BIAS_ON:
	case SND_SOC_BIAS_PREPARE:
	case SND_SOC_BIAS_STANDBY:
	case SND_SOC_BIAS_OFF:
		break;
	}

	codec->bias_level = level;
	return 0;
}


static struct snd_soc_codec *dp52_codec;

static int dp52_soc_probe(struct platform_device *pdev)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct snd_soc_codec *codec;
	int ret = 0;

	if (dp52_codec == NULL) {
		dev_err(&pdev->dev, "Codec device not registered\n");
		return -ENODEV;
	}

	codec = dp52_codec;
	socdev->card->codec = codec;

	/* register pcms */
	ret = snd_soc_new_pcms(socdev, SNDRV_DEFAULT_IDX1, SNDRV_DEFAULT_STR1);
	if (ret < 0) {
		dev_err(codec->dev, "Failed to create PCMs: %d\n", ret);
		goto pcm_err;
	}

	snd_soc_add_controls(codec, dp52_snd_controls,
		ARRAY_SIZE(dp52_snd_controls));

	snd_soc_dapm_new_controls(codec, dp52_dapm_widgets,
		ARRAY_SIZE(dp52_dapm_widgets));

	snd_soc_dapm_add_routes(codec, dp52_dapm_paths,
		ARRAY_SIZE(dp52_dapm_paths));

	snd_soc_dapm_new_widgets(codec);

	return 0;

pcm_err:
	return ret;
}

static int dp52_soc_remove(struct platform_device *pdev)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);

	snd_soc_free_pcms(socdev);
	snd_soc_dapm_free(socdev);
	return 0;
}


struct snd_soc_codec_device soc_codec_dev_dp52 = {
	.probe =	dp52_soc_probe,
	.remove =	dp52_soc_remove,
};
EXPORT_SYMBOL(soc_codec_dev_dp52);


static int __devinit dp52_codec_register(struct dp52_priv *dp52)
{
	struct dp52_codec_data *pdata = dev_get_platdata(dp52->codec.dev);
	struct snd_soc_codec *codec = &dp52->codec;
	int ret;
	//u16 reg;

	if (dp52_codec) {
		dev_err(codec->dev, "Another DP52 CODEC driver is registered\n");
		return -EBUSY;
	}

	if (!pdata) {
		dev_warn(codec->dev, "No platform data supplied\n");
	} else {
		/* TODO: Extract configuration data */
	}

	mutex_init(&codec->mutex);
	INIT_LIST_HEAD(&codec->dapm_widgets);
	INIT_LIST_HEAD(&codec->dapm_paths);

	snd_soc_codec_set_drvdata(codec, dp52);
	codec->name = "DP52";
	codec->owner = THIS_MODULE;
	codec->bias_level = SND_SOC_BIAS_OFF;
	codec->set_bias_level = dp52_set_bias_level;
	codec->dai = &dp52_dai;
	codec->num_dai = 1;
	codec->write = dp52_write;
	codec->read = dp52_read;
	codec->reg_cache_step = 2;
	codec->reg_cache_size = sizeof(dp52->reg_cache);
	codec->reg_cache = &dp52->reg_cache;

	dp52_dai.dev = codec->dev;

	/* TODO: Initialize register cache? */

	/* TODO: Initialize CODEC */

	dp52_codec = codec;

	ret = snd_soc_register_codec(codec);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to register codec: %d\n", ret);
		goto err;
	}

	ret = snd_soc_register_dai(&dp52_dai);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to register DAI: %d\n", ret);
		goto err_codec;
	}

	return 0;

err_codec:
	snd_soc_unregister_codec(codec);
err:
	dp52_codec = NULL;
	return ret;
}

static void __devexit dp52_codec_unregister(struct dp52_priv *dp52)
{
	snd_soc_unregister_dai(&dp52_dai);
	snd_soc_unregister_codec(&dp52->codec);
	dp52_codec = NULL;
}

static int __devinit dp52_codec_probe(struct platform_device *pdev)
{
	struct dp52_priv *dp52;
	struct snd_soc_codec *codec;
	int ret;

	dp52 = kzalloc(sizeof(struct dp52_priv), GFP_KERNEL);
	if (dp52 == NULL) {
		return -ENOMEM;
	}

	codec = &dp52->codec;
	codec->dev = &pdev->dev;
	platform_set_drvdata(pdev, dp52);

	ret = dp52_codec_register(dp52);
	if (ret) {
		kfree(dp52);
	}

	return ret;
}

static int __devexit dp52_codec_remove(struct platform_device *pdev)
{
	struct dp52_priv *dp52 = platform_get_drvdata(pdev);
	dp52_codec_unregister(dp52);
	kfree(dp52);
	return 0;
}

static struct platform_driver dp52_codec_driver = {
	.probe = dp52_codec_probe,
	.remove = __devexit_p(dp52_codec_remove),
	.driver = {
		.name  = "dp52-codec",
		.owner = THIS_MODULE,
	},
};

static int __init dp52_modinit(void)
{
	return platform_driver_register(&dp52_codec_driver);
}
module_init(dp52_modinit);

static void __exit dp52_modexit(void)
{
	platform_driver_unregister(&dp52_codec_driver);
}
module_exit(dp52_modexit);

MODULE_DESCRIPTION("ASoC DP52 driver");
MODULE_AUTHOR("Jan Kloetzke, Avi Miller");
MODULE_LICENSE("GPL");


