//**************************************************************************************************
//                                        PrcNetLstr.cpp                                           *
//                                       ----------------                                          *
// Started     : 2004-01-28                                                                        *
// Last Update : 2023-05-27                                                                        *
// Copyright   : (C) 2004-2023 MSWaters                                                            *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "PrcNetLstr.hpp"

//**************************************************************************************************
// Allocate storage for static data members.

wxArrayString  PrcNetLstr::m_oasGuileProcs;

//**************************************************************************************************
// Constructor.

PrcNetLstr::PrcNetLstr( void ) : PrcBase( wxPROCESS_REDIRECT )
{
  // Load the Guile procedure array with some default values
  m_oasGuileProcs.Add( "" );
  m_oasGuileProcs.Add( PRCNETLST_GUILE_PROC );
  m_szGuileProc = 0;

  // Clear the object attributes
  bClear( );

  // Set the log file name
  bSetLogFile( PRCNETLST_LOG_FILE );

#ifndef TEST_APPPRCNETLIST
  // Attempt to set and find the binary file
  bSetEdaToolSuite( g_oConfig.eGetEdaToolSuite( ) );

  // Set the Guile procedure
  bSetGuileProc( g_oConfig.rosGetGuileProc( ) );
#endif // TEST_PRCNETLIST
}

//**************************************************************************************************
// Destructor.

PrcNetLstr::~PrcNetLstr( )
{
}

//**************************************************************************************************
// Clear the object attributes.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcNetLstr::bClear( void )
{
  bool  bRtn=true;

  if( bSetGuileProc ( "" ) ) bRtn = false;
  if( bSetSchemFiles( "" ) ) bRtn = false;
  if( bSetNetLstFile( "" ) ) bRtn = false;

  return( bRtn );
}

//**************************************************************************************************
// Initialize the array of possible Guile procedure names.

void  PrcNetLstr::InitGuileProcs( void )
{
  SysScan            oSysScan;
  wxArrayString      oas1;
  wxString           osGProc, os1;
  wxStringTokenizer  ostk1;
  size_t             sz1;

  // Temporarily record the currently selected Guile procedure
  osGProc = m_oasGuileProcs.Item( m_szGuileProc );

  m_oasGuileProcs.Clear( );   // Clear the list of possible Guile procedures
  m_oasGuileProcs.Add( "" );  // First location is the default and is deliberately left empty

  // Get a list of Guile procedures from the netlister utility :
  //     gnetlist --list-backends
  //   or
  //     lepton-netlist --list-backends
  os1 = rofnGetBinFile( ).GetFullPath( ) + " --list-backends -L " + oSysScan.rosGetPathInstall( )
      + "/sch ";
  if( wxExecute( os1, oas1 ) == 0 )
  {
    // The string array contains the list of Guile procedures
    if( oas1.GetCount( ) > 1 )
    {
      for( sz1=1; sz1<oas1.GetCount( ); sz1++ )
      {
        os1 = oas1.Item( sz1 );

        // The two netlister use different output formats, only want the last field
        if( eGetEdaToolSuite( ) == eEDA_GEDAGAF ) os1 = os1.AfterLast( wxUniChar( '\t' ) );

        // Only want SPICE related Guile procedures
        if( os1.Find( "spice-msw" ) != wxNOT_FOUND ) m_oasGuileProcs.Add( os1 );
        if( os1.Find( "spice-sdb" ) != wxNOT_FOUND ) m_oasGuileProcs.Add( os1 );
      }
    }
  }
  else m_oasGuileProcs.Add( PRCNETLST_GUILE_PROC ); // Default Guile procedure

  // Now attempt to reinstate the previous Guile procedure selection otherwise use the default
  if( ! bSetGuileProc( osGProc ) ) bSetGuileProc( PRCNETLST_GUILE_PROC );
}

//**************************************************************************************************
// Set the EDA tool suite..
//
// Argument List :
//   eEDA_TS - The enumerated type specifier for the desired tool suite
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcNetLstr::bSetEdaToolSuite( eTypeEDA eEDA_TS )
{
  bool  bRtn;

  switch( eEDA_TS )
  {
    case eEDA_LEPTON  : bRtn = bSetBinFile( BIN_LNETLIST ); break;
    case eEDA_GEDAGAF : bRtn = bSetBinFile( BIN_GNETLIST ); break;
    default           : bRtn = FALSE;
  }

  InitGuileProcs( );

  return( bRtn );
}

//**************************************************************************************************
// Set the Guile procedure to be used to import the schematic file.
//
// Argument List :
//   rosGProc - The Guile procedure name
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcNetLstr::bSetGuileProc( const wxString & rosGProc )
{
  int  i1;

  if( m_oasGuileProcs.IsEmpty( ) ) return( false );

  i1 = m_oasGuileProcs.Index( rosGProc );
  if( i1 == wxNOT_FOUND )          return( false );

  m_szGuileProc = (size_t) i1;

  return( true );
}

//**************************************************************************************************
// Set the schematic file name/s.
//
// Argument List :
//   rosFileNames - A string containing the full path and file name/s
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcNetLstr::bSetSchemFiles( const wxString & rosFileNames )
{
  wxStringTokenizer  ostk1;
  wxArrayString      osa1;

  ostk1.SetString( rosFileNames );
  while( ostk1.HasMoreTokens( ) ) osa1.Add( ostk1.GetNextToken( ) );

  return( bSetSchemFiles( osa1 ) );
}

//**************************************************************************************************
// Set the schematic file name/s.
//
// Argument List :
//   roasFileNames - A string array containing the full path and file name/s
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcNetLstr::bSetSchemFiles( const wxArrayString & roasFileNames )
{
  wxFileName  ofn1;
  size_t      sz1;

  // Clear the current list of schematic files
  m_oaNameSchems.Clear( );

  // Check the argument is empty
  if( roasFileNames.IsEmpty( ) )                                return( true );

  // Add the new schematic file name/s to the list
  for( sz1=0; sz1<roasFileNames.GetCount( ); sz1++ )
  {
    ofn1 = roasFileNames.Item( sz1 );
    if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( "." );

    if( ! ofn1.IsOk( )       ) continue;
    if( ! ofn1.FileExists( ) ) continue;

    m_oaNameSchems.Add( ofn1 );
  }

  // Check that at least one schematic file name was accepted
  if( m_oaNameSchems.IsEmpty( ) )                               return( false );

  // Set the log file path
  ofn1 = rofnGetLogFile( );
  ofn1.SetPath( m_oaNameSchems.Item( 0 ).GetPath( ) );
  bSetLogFile( ofn1.GetFullPath( ) );

  // Check if any of the schematic files were invalid
  if( m_oaNameSchems.GetCount( ) != roasFileNames.GetCount( ) ) return( false );

  return( true );
}

//**************************************************************************************************
// Set the full netlist file name.
//
// Argument List :
//   rosFileName - A string containing the full path and file name
//                 (If rosFName = "use-schem-name" the netlist name is to be generated from the
//                  schematic file name)
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcNetLstr::bSetNetLstFile( const wxString & rosFileName )
{
  wxFileName  ofn1;

  // Set the netlist name
  if( ! rosFileName.IsEmpty( ) )
  {
    if( rosFileName == PRCNETLST_USE_SCHEM )
    { // Generate the netlist file name from the schematic file name
      ofn1 = rofnGetSchemFile( );
      ofn1.SetExt( "ckt" );
    }
    else ofn1 = rosFileName;

    // Set the netlist path if it hasn't been set
    if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( "." );
    ofn1.Normalize( wxPATH_NORM_DOTS | wxPATH_NORM_TILDE | wxPATH_NORM_ABSOLUTE |
                    wxPATH_NORM_LONG | wxPATH_NORM_SHORTCUT );  // Expand "..", "~', etc.

    // Check that the file name is OK
    if( ! ofn1.IsOk( ) ) return( false );
  }
  else m_ofnNameNetList.Clear( );

  // Set the netlist file path and name
  m_ofnNameNetList = ofn1;

  // Set the log file path
  ofn1 = rofnGetLogFile( );
  ofn1.SetPath( m_ofnNameNetList.GetPath( ) );
  bSetLogFile( ofn1.GetFullPath( ) );

  return( true );
}

//**************************************************************************************************
// Get the EDA tool suite.
//
// Return Values :
//   Success - The enumerate type for the EDA tool suite
//   Failure - eEDA_NONE

eTypeEDA  PrcNetLstr::eGetEdaToolSuite( void )
{
  wxString  os1;

  os1 = rofnGetBinFile( ).GetFullName( );

  if( os1 == BIN_LNETLIST ) return( eEDA_LEPTON  );
  if( os1 == BIN_GNETLIST ) return( eEDA_GEDAGAF );
  else                      return( eEDA_NONE    );
}

//**************************************************************************************************
// Get a Guile procedure from the list of procedure names.
//
// Argument List :
//   szIndex - A zero based index to the procedure name
//             (If szIndex = CUR_SEL_PROC return currently selected procedure)
//
// Return Values :
//   Success - The procedure name
//   Failure - An empty string

const wxString & PrcNetLstr::rosGetGuileProc( size_t szIndex )
{
  static  wxString  osEmpty=wxEmptyString;

  if( szIndex == PRCNETLST_CUR_PROC ) szIndex = m_szGuileProc;

  if( szIndex >= m_oasGuileProcs.GetCount( ) ) return( osEmpty );

  return( m_oasGuileProcs[ szIndex ] );
}

//**************************************************************************************************
// Get a schematic file from the list of schematic files.
//
// Argument List :
//   szIndex - A zero based index to the file name
//
// Return Values :
//   Success - The procedure name
//   Failure - An empty file name

const wxFileName & PrcNetLstr::rofnGetSchemFile( size_t szIndex )
{
  static  wxFileName  ofnEmpty;

  if( szIndex >= m_oaNameSchems.GetCount( ) ) return( ofnEmpty );

  return( m_oaNameSchems[ szIndex ] );
}

//**************************************************************************************************
// Make a netlist file from a schematic file.
//
// (Eg. using the following: gnetlist -v -g spice-sdb -o test.ckt test.sch)
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcNetLstr::bExec( void )
{
  wxString    osArgLst;
  SysScan     oSysScan;
  wxFileName  ofn1;
  wxString    os1;
  size_t      sz1;

  // Check that the schematic and netlist files have been set
  if( m_oaNameSchems.IsEmpty( ) ) return( false );
  if( ! m_ofnNameNetList.IsOk( ) )
    if( ! bSetNetLstFile( ) )     return( false );

  // Set the various gnetlist modes
  if(   g_oConfig.bGetVerboseMode( ) ) osArgLst << "--verbose ";
  if( ! g_oConfig.bGetIncludeMode( ) ) osArgLst << "-O include_mode ";
  if(   g_oConfig.bGetEmbedMode( )   ) osArgLst << "-O embedd_mode ";
  if(   g_oConfig.bGetNoMungeMode( ) ) osArgLst << "-O nomunge_mode ";
  osArgLst << "-O sort_mode ";

  // Added a directory to search for Guile procedure backends
  osArgLst << "-L " << oSysScan.rosGetPathInstall( ) << "/sch ";

  // Specify the guile procedure name to be used
  os1 = m_oasGuileProcs.Item( m_szGuileProc );
  if( os1.IsEmpty( ) ) os1 = PRCNETLST_GUILE_PROC;
  osArgLst << "-g " << os1 << ' ';

  // Append output file name
#ifndef __WXMSW__
  osArgLst << "-o " << rosEscSpaceChrs( m_ofnNameNetList.GetFullPath( ) );
#else
  os1 = m_ofnNameNetLst.GetFullPath( );
  os1.Replace( "\\", "/", true );
  osArgLst << "-o " << os1;
#endif // __WXMSW__

  // Append input file name/s
#ifndef __WXMSW__
  for( sz1=0; sz1<m_oaNameSchems.GetCount( ); sz1++ )
    osArgLst << " " << rosEscSpaceChrs( m_oaNameSchems.Item( sz1 ).GetFullPath( ) );
#else
  osArgLst << " -- ";  // Treat all remaining arguments as schematic filenames
  for( sz1=0; sz1<m_oaNameSchems.GetCount( ); sz1++ )
  {
    os1 = m_oaNameSchems.Item( sz1 ).GetFullPath( );
    os1.Replace( "\\", "/", true );
    osArgLst << ' ' << os1;
  }
#endif // __WXMSW__

  // Set the argument list
  bSetArgLst( osArgLst );

  // Set the working directory
  bSetCwd( rofnGetSchemFile( ).GetPath( ) );

  // Execute the process
  if( ! PrcBase::bExecAsync( ) )  return( false );

  // Capture the process output
  if( ! bLogOutput( ) )           return( false );

  return( PrcBase::bIsOk( ) );
}

//**************************************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  PrcNetLstr::Print( const wxString & rosPrefix )
{
  size_t  sz1;

  PrcBase::Print( rosPrefix + "PrcBase::" );

  // size_t  m_szGuileProc
  std::cout << rosPrefix.mb_str( ) << "m_szGuileProc      : " << m_szGuileProc << '\n';

  // wxArrayString  m_oasGuileProcs
  std::cout << rosPrefix.mb_str( ) << "m_oasGuileProcs[ ] : ";
  if( ! m_oasGuileProcs.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_oasGuileProcs.GetCount( ); sz1++ )
    {
      if( sz1 > 0 ) std::cout << rosPrefix.mb_str( ) << "                     ";
      std::cout << m_oasGuileProcs.Item( sz1 ).mb_str( ) << '\n';
    }
  }
  else std::cout << '\n';

  // ArrayFileName  m_oaNameSchems
  std::cout << rosPrefix.mb_str( ) << "m_oaNameSchems[ ]  : ";
  if( ! m_oaNameSchems.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_oaNameSchems.GetCount( ); sz1++ )
    {
      if( sz1 > 0 ) std::cout << rosPrefix.mb_str( ) << "                     ";
      std::cout << m_oaNameSchems.Item( sz1 ).GetFullPath( ).mb_str( ) << '\n';
    }
  }
  else std::cout << '\n';

  // wxFileName  m_ofnNameNetLst
  std::cout << rosPrefix.mb_str( ) << "m_ofnNameNetList   : "
            << m_ofnNameNetList.GetFullPath( ).mb_str( ) << '\n';
}

//**************************************************************************************************
